#pragma once

#include "settings.h"
#include "../mod_log/mod_log.h"
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>

#include "processor_def.h"

namespace yxiva::mailpusher {

template <typename HttpClient>
struct searchapp_sender
{
    using yield_context_t = yplatform::yield_context<searchapp_sender<HttpClient>>;

    struct notification
    {
        string event_name;
        string session_key;
        string body;
    };

    shared_ptr<task> task;
    ymod_ratecontroller::rate_controller_ptr rate_controller;
    HttpClient& http_client;
    const settings& settings;
    callback_t cb;

    ymod_ratecontroller::completion_handler rate_controller_on_complete{};
    error_code error{};
    struct event* event = nullptr;
    size_t fresh_counter = 0;
    std::vector<string> uuids{};

    static const string& name()
    {
        static const string NAME = "searchapp_send";
        return NAME;
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        yhttp::response resp = {})
    {
        try
        {
            reenter(yield_context)
            {
                yield rate_controller->post(yield_context, "", task->deadline());

                if (!settings.searchapp_installations.count(task->uid))
                {
                    yield break;
                }

                if (error = error::from_rate_controller_error(ec))
                {
                    yield break;
                }

                if (!find_corresponding_event())
                {
                    yield break;
                }

                if (!find_corresponding_uuids())
                {
                    yield break;
                }

                yield do_request(yield_context);
                if (auto err = error::from_http_response(ec, resp))
                {
                    if (ec)
                    {
                        LOG_ERROR("sup_http_error", ec.message());
                    }
                    else
                    {
                        LOG_ERROR("sup_http_response", "code " + std::to_string(resp.status));
                    }
                }
                else
                {
                    json_value resp_json;
                    if (auto res = json_parse(resp_json, resp.body))
                    {
                        task->sup_id = resp_json["id"].to_string();
                    }
                    else
                    {
                        LOG_ERROR("sup_http_response_json", res.error_reason);
                    }
                }
            }
        }
        catch (const std::exception& ex)
        {
            LOG_ERROR("exception", ex.what());
            error = make_error(error::internal_error);
        }

        if (yield_context.is_complete())
        {
            cb(error);
            yplatform::safe_call(rate_controller_on_complete);
        }
    }

    void do_request(yield_context_t yield_context)
    {
        json_value body;
        json_value receiver(json_type::tarray);
        for (auto&& uuid : uuids)
        {
            receiver.push_back("uuid:" + uuid);
        }
        body["receiver"] = receiver;
        json_value notification;
        notification["title"] = "Почта";
        notification["body"] = text_declined(fresh_counter);
        body["notification"] = notification;
        body["ttl"] = 86400;
        body["project"] = "mail";
        json_value throttle_policies;
        throttle_policies["install_id"] = "yamail";
        throttle_policies["device_id"] = "yamail";
        throttle_policies["device_id"] = "yamail";
        body["throttle_policies"] = throttle_policies;
        json_value custom_data;
        custom_data["push_id"] = "mail.new_message_for_interested";
        custom_data["push_action"] = "uri";
        custom_data["push_uri"] = "https://mail.yandex.ru?transit_id=" + task->uniq_id();
        custom_data["topic_push"] = "mail_push";
        custom_data["transit_id"] = task->uniq_id();
        body["data"] = custom_data;

        http_client.async_run(
            task,
            yhttp::request::POST(
                settings.searchapp.request,
                settings.searchapp.auth_header + "Content-Type: application/json;charset=UTF-8\r\n",
                std::move(json_write(body))),
            yield_context);
    }

    // Rate controller handler.
    void operator()(
        yield_context_t yield_context,
        const error_code& ec,
        ymod_ratecontroller::completion_handler on_complete)
    {
        rate_controller_on_complete = std::move(on_complete);
        (*this)(yield_context, ec);
    }

    bool find_corresponding_event()
    {
        for (auto it = task->events.rbegin(); it != task->events.rend(); it++)
        {
            auto&& fresh_count_json = it->args["fresh_count"];
            if (fresh_count_json.is_uint64() && fresh_count_json.to_uint64())
            {
                event = &*it;
                fresh_counter = fresh_count_json.to_uint64();
                return true;
            }
        }
        return false;
    }

    bool find_corresponding_uuids()
    {
        auto it = settings.searchapp_installations.find(task->uid);
        if (it == settings.searchapp_installations.end()) return false;
        auto&& installations = it->second;
        uuids.reserve(installations.size());
        for (auto&& i : installations)
        {
            // Filter devices with subscriptions in Xiva.
            bool found = false;
            for (auto&& j : task->subscriptions)
            {
                // Installations' device IDs are converted while reading settings.
                if (auto* mobile_sub = j.mobile())
                {
                    if (i.second == canonize_device_id(mobile_sub->device))
                    {
                        YLOG_CTX_GLOBAL(task, info)
                            << "searchapp_sender: ignoring duplicated device " << i.second;
                        found = true;
                    }
                }
            }
            if (!found) uuids.push_back(i.first);
        }
        return uuids.size();
    }

    string text_declined(size_t number)
    {
        string result = "У вас " + std::to_string(number);
        if ((number % 100) / 10 == 1)
        {
            result += " новых писем";
        }
        else if (number % 10 == 1)
        {
            result += " новое письмо";
        }
        else if (number % 10 >= 2 && number % 10 <= 4)
        {
            result += " новых письма";
        }
        else
        {
            result += " новых писем";
        }
        return result;
    }
};

}

#include "processor_undef.h"