#pragma once

#include <mailpusher/types.h>
#include <yxiva/core/subscriptions.h>
#include <ymod_httpclient/cluster_client.h>
#include <yplatform/ptree.h>
#include <yplatform/util/sstream.h>
#include <set>
#include <unordered_map>
#include <boost/algorithm/string.hpp>

namespace yxiva::mailpusher {
namespace detail {

inline string read_and_trim_file(const string& file_name)
{
    std::ifstream fs(file_name);
    if (!fs.is_open() || !fs.good()) throw std::runtime_error("failed to open file " + file_name);
    string res(std::istream_iterator<char>(fs >> std::noskipws), std::istream_iterator<char>());
    if (!fs.eof()) throw std::runtime_error("failed to read file " + file_name);
    boost::trim(res);
    return res;
}

inline string get_secret(const yplatform::ptree& conf, const string& name = "secret")
{
    string secret = conf.get<string>(name, "");
    if (secret.empty()) secret = read_and_trim_file(conf.get<string>(name + "_file"));
    return secret;
}

template <typename Settings>
void read_http_settings(Settings& node, const yplatform::ptree& conf)
{
    if (auto http_conf = conf.get_child_optional("http"))
    {
        node.http.parse_ptree(*http_conf);
    }
}

template <typename Settings>
void read_basic_settings(Settings& node, const yplatform::ptree& conf)
{
    node.request = conf.get<string>("request");
    read_http_settings(node, conf);
    if (node.http.nodes.empty())
    {
        node.http.nodes = { conf.get<string>("host") };
    }
}

template <typename Settings>
void read_basic_settings(Settings& node, const yplatform::ptree& conf, const string& name)
{
    read_basic_settings(node, conf.get_child(name));
}

inline string get_auth_header(const yplatform::ptree& conf, const string& path)
{
    string auth;
    if (auto auth_conf = conf.get_child_optional(path))
    {
        yplatform::sstream(auth) << "Authorization: " << auth_conf->get<string>("scheme") << " "
                                 << detail::get_secret(*auth_conf) + "\r\n";
    }
    return auth;
}

}

struct settings
{
    struct list
    {
        string request;
        yhttp::cluster_client::settings http;
    };
    std::map<string, struct list> list;

    struct counters
    {
        string request;
        yhttp::cluster_client::settings http;
    };
    std::map<string, struct counters> counters;

    struct
    {
        string request;
        yhttp::cluster_client::settings http;
        size_t max_mids_in_filter_search_request = 0;
        unsigned fallback_to_db_master_max_lag = 0;
    } meta;

    struct
    {
        string request;
        yhttp::cluster_client::settings http;
        unsigned max_emails_in_profiles_request = 0;
        unsigned disabled_percent = 0;
    } ava;

    struct
    {
        string request;
        yhttp::cluster_client::settings http;
        unsigned ttl = 0;
        unsigned rtec_3674_rollout_percent = 0;
        std::set<int> ignored_folder_codes;
    } send;

    struct
    {
        string request;
        yhttp::cluster_client::settings http;
        string auth_header;
    } searchapp;

    // Mapping uid -> (uuid, device).
    std::unordered_map<string, std::vector<std::pair<string, string>>> searchapp_installations;
    string environment;

    settings() = default; // for tests

    settings(const yplatform::ptree& conf)
    {
        read_all_environments(list, conf, "xiva_list");
        read_all_environments(counters, conf, "counters");
        detail::read_basic_settings(meta, conf, "meta");
        detail::read_basic_settings(ava, conf, "ava");
        detail::read_basic_settings(send, conf, "xiva_send");
        detail::read_basic_settings(searchapp, conf, "searchapp_send");

        searchapp.auth_header = detail::get_auth_header(conf, "searchapp_send.auth");
        meta.max_mids_in_filter_search_request =
            conf.get<size_t>("meta.max_mids_in_filter_search_request");
        meta.fallback_to_db_master_max_lag =
            conf.get<unsigned>("meta.fallback_to_db_master_max_lag");
        ava.max_emails_in_profiles_request =
            conf.get<unsigned>("ava.max_emails_in_profiles_request");
        ava.disabled_percent = conf.get<unsigned>("ava.disabled_percent");
        send.ttl = conf.get<unsigned>("xiva_send.ttl", 604800);
        send.rtec_3674_rollout_percent =
            conf.get<unsigned>("xiva_send.rtec_3674_rollout_percent", 0);
        yplatform::read_ptree(
            send.ignored_folder_codes, conf.get_child("xiva_send"), "ignored_folder_codes");
        environment = detail::read_and_trim_file(
            conf.get<string>("environment_file", "/etc/environment.name"));

        auto path = conf.get_optional<string>("searchapp_installations_file");
        if (path)
        {
            std::ifstream ifs(*path);
            if (ifs.is_open())
            {
                string uid, uuid, device;
                while (std::getline(ifs, uid) && std::getline(ifs, uuid) &&
                       std::getline(ifs, device))
                {
                    device = canonize_device_id(device);
                    searchapp_installations[uid].push_back({ uuid, device });
                }
            }
            else
            {
                YLOG_LOCAL(error) << "failed to open '" << *path << "'";
            }
        }

        for (auto&& it : boost::make_iterator_range(conf.equal_range("searchapp_installations")))
        {
            auto uid = it.second.get<string>("uid");
            auto uuid = it.second.get<string>("uuid");
            auto device = canonize_device_id(it.second.get<string>("device"));
            searchapp_installations[uid].push_back({ uuid, device });
        }

        if (searchapp_installations.size())
        {
            YLOG_LOCAL(info) << "loaded searchapp_installations for "
                             << searchapp_installations.size() << " users";
        }
    }

private:
    template <typename Settings>
    void read_all_environments(
        std::map<string, Settings>& nodes,
        const yplatform::ptree& conf,
        const string& name)
    {
        for (auto& [environment, sub_conf] : conf.get_child(name))
        {
            YLOG_LOCAL(info) << "reading " << name << " settings for " << environment;
            detail::read_basic_settings(nodes[environment], sub_conf);
        }
    }
};

}
