#pragma once

#include "subscription_fetcher_impl.h"
#include "settings.h"
#include "../mod_log/mod_log.h"
#include "parse_accounts.h"
#include <mailpusher/task.h>
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>

#include "processor_def.h"

namespace yxiva::mailpusher {

template <typename HttpClient>
struct subscription_fetcher
{
    using yield_context_t = yplatform::yield_context<subscription_fetcher>;
    using impl_t = subscription_fetcher_impl<HttpClient>;

    shared_ptr<task> task;
    rate_controllers<ymod_ratecontroller::rate_controller_ptr> rate_controllers;
    http_clients<HttpClient&> http_clients;
    const settings& settings;
    callback_t cb;

    std::unordered_map<string, device>::iterator device_it = {};
    std::size_t account_index = 0;

    static const string& name()
    {
        return impl_t::name();
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        const std::vector<subscription>& subscriptions = {})
    {
        try
        {
            reenter(yield_context)
            {
                yield list_subscriptions(task->uid, settings.environment, yield_context);
                if (!ec)
                {
                    task->subscriptions = subscriptions;
                }
                parse_devices();
                // Check whether accounts are still logged in on devices.
                device_it = task->devices.begin();
                for (; device_it != task->devices.end(); ++device_it)
                {
                    account_index = 0;
                    for (; account_index < device_it->second.accounts.size(); ++account_index)
                    {
                        yield list_subscriptions(next_unprocessed_account(), yield_context);
                        if (!ec)
                        {
                            next_unprocessed_account().has_subscriptions =
                                subscribed_on_same_device(device_it->second, subscriptions);
                        }
                    }
                }
                cb(ec);
            }
        }
        catch (const std::exception& ex)
        {
            LOG_ERROR("exception", ex.what());
            cb(make_error(error::internal_error));
        }
    }

    void operator()(typename yield_context_t::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            LOG_ERROR("exception", e.what());
        }
        yplatform::safe_call(cb, make_error(error::internal_error));
    }

private:
    void list_subscriptions(const account& account, yield_context_t yield_context)
    {
        if (account.uid == task->uid && account.environment == settings.environment)
        {
            operator()(yield_context, {}, task->subscriptions);
        }
        else
        {
            list_subscriptions(account.uid, account.environment, yield_context);
        }
    }

    void list_subscriptions(
        const string& uid,
        const string& environment,
        yield_context_t yield_context)
    {
        yplatform::spawn(impl_t{ task,
                                 uid,
                                 rate_controllers.at(environment),
                                 http_clients.at(environment),
                                 settings.list.at(environment),
                                 yield_context });
    }

    bool subscribed_on_same_device(
        const device& my_device,
        const std::vector<subscription>& subscriptions)
    {
        return std::any_of(subscriptions.begin(), subscriptions.end(), [&my_device](auto& s) {
            return s.mobile() && s.mobile()->platform == my_device.platform &&
                s.mobile()->device == my_device.id && s.mobile()->app == my_device.app;
        });
    }

    void parse_devices()
    {
        for (auto& subscription : task->subscriptions)
        {
            if (!subscription.mobile())
            {
                continue;
            }

            std::vector<account> accounts;
            if (!parse_accounts(subscription.common()->extra, accounts))
            {
                continue;
            }

            task->devices.emplace(
                subscription.common()->id,
                device{ subscription.mobile()->device,
                        subscription.mobile()->platform,
                        subscription.mobile()->app,
                        std::move(accounts) });
        }
    }

    auto& next_unprocessed_account()
    {
        return device_it->second.accounts.at(account_index);
    }
};

}

#include "processor_undef.h"