#pragma once

#include "settings.h"
#include "../mod_log/mod_log.h"
#include <mailpusher/types.h>
#include <mailpusher/errors.h>
#include <yplatform/coroutine.h>
#include <yplatform/util/safe_call.h>

#include "processor_def.h"

namespace yxiva::mailpusher {

template <typename HttpClient>
struct subscription_fetcher_impl
{
    using yield_context_t = yplatform::yield_context<subscription_fetcher_impl>;
    using subscriptions_callback_t =
        std::function<void(const error_code&, const std::vector<subscription>&)>;

    yplatform::task_context_ptr ctx;
    string uid;
    ymod_ratecontroller::rate_controller_ptr rate_controller;
    HttpClient& http_client;
    const struct settings::list& settings;
    subscriptions_callback_t cb;

    ymod_ratecontroller::completion_handler rate_controller_on_complete{};
    error_code error{};
    std::vector<subscription> subscriptions{};

    static const string& name()
    {
        static const string NAME = "xiva_list";
        return NAME;
    }

    void operator()(
        yield_context_t yield_context,
        const error_code& ec = {},
        yhttp::response resp = {})
    {
        reenter(yield_context)
        {
            yield rate_controller->post(yield_context, "", ctx->deadline());
            if (error = error::from_rate_controller_error(ec))
            {
                yield break;
            }
            yield do_request(yield_context);
            if (error = error::from_http_response(ec, resp))
            {
                if (ec)
                {
                    LOG_ERROR_CTX(ctx, "http_error", ec.message());
                }
                else
                {
                    LOG_ERROR_CTX(ctx, "http_response", "code " + std::to_string(resp.status));
                }
            }
            else
            {
                error = process_response(resp);
            }
        }

        if (yield_context.is_complete())
        {
            cb(error, subscriptions);
            yplatform::safe_call(rate_controller_on_complete);
        }
    }

    void operator()(typename yield_context_t::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            LOG_ERROR_CTX(ctx, "exception", e.what());
        }
        yplatform::safe_call(cb, make_error(error::internal_error), subscriptions);
        yplatform::safe_call(rate_controller_on_complete);
    }

    void do_request(yield_context_t yield_context)
    {
        http_client.async_run(
            ctx,
            yhttp::request::GET(
                settings.request + yhttp::url_encode({ { "service", "mail" }, { "user", uid } })),
            yield_context);
    }

    // Rate controller handler.
    void operator()(
        yield_context_t yield_context,
        const error_code& ec,
        ymod_ratecontroller::completion_handler on_complete)
    {
        rate_controller_on_complete = std::move(on_complete);
        (*this)(yield_context, ec);
    }

    error_code process_response(const yhttp::response& resp)
    {
        json_value json_list;
        if (auto res = json_parse(json_list, resp.body, json_type::tarray); !res)
        {
            LOG_ERROR_CTX(ctx, "malformed_response", res.error_reason);
            return make_error(error::bad_gateway);
        }
        subscriptions.reserve(json_list.size());
        for (auto&& json_sub : json_list.array_items())
        {
            auto [res, sub] = decode_subscription(json_sub);
            if (!res)
            {
                LOG_ERROR_CTX(ctx, "bad_subscription_fields", res.error_reason);
                return make_error(error::bad_gateway);
            }
            subscriptions.push_back(sub);
        }
        return make_error(error::success);
    }
};

}

#include "processor_undef.h"