#include "postgre.h"
#include <yxiva/core/types.h>
#include <yxiva/core/filter.h>
#include <yxiva/core/services/msgpack_decoder.h>
#include <yxiva/core/packing.hpp>
#include <yxiva/core/shards/static_storage.h>
#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>
#include <yplatform/find.h>
#include <ymod_pq/call.h>
#include <ymod_httpclient/cluster_client.h>
#include <pipeline/processor.h>
#include <pipeline/stream_strand.h>
#include <sstream>
#include <memory>

namespace yxiva { namespace mesh {

struct stream_holder
{
public:
    unsigned total_gids = { 0 };
    std::atomic_uint ok = { 0 };
    std::atomic_uint total = { 0 };
    std::shared_ptr<yxiva::message> message;
    ymod_webserver::http::stream_ptr stream;

    stream_holder(ymod_webserver::http::stream_ptr stream) : stream(stream)
    {
    }

    ~stream_holder()
    {
        auto summary = std::to_string(ok.load()) + "/" + std::to_string(total.load());
        YLOG_CTX_GLOBAL(stream->ctx(), info)
            << std::to_string(total_gids) + " gids with " << summary;
        stream->result(ymod_webserver::codes::ok, summary);
    }
};

using pipe_settings = pipeline::StreamSettings;
template <typename T>
using pipe_stream = pipeline::StreamStrand<T>;
template <typename T>
using pipe_processor = pipeline::Processor<T>;

namespace http_pipe {
struct data
{
    std::shared_ptr<stream_holder> holder;
    yplatform::task_context_ptr ctx;
    sub_t sub;
    yxiva::message message;
    bool success;
};
using data_ptr = std::shared_ptr<data>;

class proc : public pipe_processor<pipe_stream<data_ptr>>
{
    std::shared_ptr<yhttp::cluster_client> http_client;
    using base_t = pipe_processor<pipe_stream<data_ptr>>;

public:
    proc(
        boost::asio::io_service& io,
        const pipe_settings& st,
        std::shared_ptr<yhttp::cluster_client> http_client)
        : base_t(io, st), http_client(http_client)
    {
        input()->label("http_proc");
    }

protected:
    void on_data(stream_ptr stream, std::size_t begin_id, std::size_t end_id) override
    {
        for (std::size_t i = begin_id; i < end_id; i++)
        {
            start_http_call(stream->at(i), i);
        }
    }

    void start_http_call(data_ptr el, std::size_t i, unsigned attempt = 0)
    {
        yhttp::cluster_client::options opts;
        opts.reuse_connection = true;
        string body = yxiva::pack(el->message); // TODO OPTIMIZE store in data already packed string
        body += yxiva::pack(el->sub);
        auto url = "/" + el->sub.service +
            yhttp::url_encode({ { "uid", el->sub.uid },
                                { "event_ts", std::to_string(el->message.event_ts) },
                                { "service", el->sub.service },
                                { "uidservice", el->sub.uid + el->sub.service } });
        auto out_req =
            yhttp::request::POST(url, {}, std::move(body)); // TODO pass packed as shared_ptr
        auto shared_this = shared_from_this();
        http_client->async_run(
            el->ctx,
            out_req,
            opts,
            [shared_this, this, el, i, attempt](
                const boost::system::error_code& err, yhttp::response response) {
                if (!err && (response.status / 100) == 2)
                {
                    el->success = false;
                    YLOG_CTX_GLOBAL(el->ctx, info)
                        << "status=success code=" << response.status << log_args(el);
                    ++el->holder->ok;
                }
                else
                {
                    YLOG_CTX_GLOBAL(el->ctx, error)
                        << "status=failed err=" << err << " code=" << response.status
                        << " attempt=" << attempt << log_args(el);
                }
                input()->commit(i);
            });
    }

    string log_args(const data_ptr& el) const
    {
        string args;
        args.reserve(512);
        args += " uid=";
        args += el->sub.uid;
        args += " service=";
        args += el->sub.service;
        args += " subid=";
        args += el->sub.id;
        args += " transit_id=";
        args += el->message.transit_id;
        return args;
    }
};
}

class mod : public yplatform::module
{
public:
    void init(yplatform::ptree const& conf)
    {
        auto webserver = yplatform::find<ymod_webserver::server>("webserver");
        using ymod_webserver::transformer;
        using ymod_webserver::argument;
        using ymod_webserver::optional_argument;
        webserver->bind("", { "/ping" }, [](ymod_webserver::http::stream_ptr stream) {
            stream->result(ymod_webserver::codes::ok, "pong");
        });
        webserver->bind(
            "",
            { "/send" },
            [this](ymod_webserver::http::stream_ptr stream, unsigned gid_from, unsigned gid_to) {
                auto message = std::make_shared<yxiva::message>();
                yxiva::services::msgpack_decoder decoder("");
                try
                {
                    decoder.decode(stream->request(), *message);
                }
                catch (const std::exception& e)
                {
                    YLOG_CTX_LOCAL(stream->ctx(), error)
                        << "message decode exception: " << e.what();
                    ymod_webserver::default_answers::send_bad_request(stream, "invalid message");
                    return;
                }
                this->send(stream, gid_from, gid_to, message);
            },
            transformer(argument<unsigned>("gid_from"), argument<unsigned>("gid_to")));
        webserver->bind(
            "",
            { "/send_test" },
            [this](
                ymod_webserver::http::stream_ptr stream,
                unsigned gid_from,
                unsigned gid_to,
                const string& uid,
                const string& service,
                const string& text) {
                auto message = std::make_shared<yxiva::message>();
                message->service = service;
                message->transit_id = stream->request()->ctx()->uniq_id();
                message->raw_data = text;
                message->uid = uid;
                this->send(stream, gid_from, gid_to, message);
            },
            transformer(
                argument<unsigned>("gid_from"),
                argument<unsigned>("gid_to"),
                optional_argument<string>("uid", ""),
                argument<string>("service"),
                optional_argument<string>("text", "")));

        db_xtable = yplatform::find<ymod_pq::call, std::shared_ptr>("pq");
        http_client = yplatform::find<yhttp::cluster_client, std::shared_ptr>("http_client");
        list_chunk_size_ = conf.get<unsigned>("list_chunk_size");

        {
            pipe_settings pipe_st(conf.get<unsigned>("pipe.http.capacity"));
            pipe_st.set_window(conf.get<unsigned>("pipe.http.parallel"));
            pipe_st.set_grouped_flush_size(conf.get<unsigned>("pipe.http.flush_by"));
            auto reactor =
                yplatform::global_reactor_set->get(conf.get<string>("pipe.http.reactor"));
            http_proc_ = std::make_shared<http_pipe::proc>(*reactor->io(), pipe_st, http_client);
        }

        shards_ = yplatform::find<yxiva::shard_config::storage, std::shared_ptr>("shards");

        // {
        //   pipe_settings pipe_st(conf.get<unsigned>("pipe.update.capacity"));
        //   pipe_st.set_window(conf.get<unsigned>("pipe.update.parallel"));
        //   pipe_st.set_grouped_flush_size(conf.get<unsigned>("pipe.update.flush_by"));
        //   auto reactor =
        //   yplatform::global_reactor_set->get(conf.get<string>("pipe.update.reactor"));
        //   update_proc_ = std::make_shared<update_pos_pipe::proc>(*reactor->io(), pipe_st,
        //   db_xtable);
        // }
        // UPDATE POSITION
        // http_proc_->input()->reset_commit_handler([this](std::shared_ptr<std::vector<http_pipe::data_ptr>>
        // data_pack) {
        //   L_(info) << data_pack->size();
        //   auto q = std::make_shared<std::vector<update_pos_pipe::data_ptr>>(data_pack->size());
        //   int current_gid = -1;
        //   auto to_insert = std::make_shared<update_pos_pipe::data>();
        //   for (auto data : *data_pack) {
        //       if (current_gid != -1 && current_gid != data->gid) {
        //           current_gid = data->gid;
        //           update_proc_->input()->put_range(q);
        //           q =
        //           std::make_shared<std::vector<update_pos_pipe::data_ptr>>(data_pack->size());
        //       }
        //       q->push_back(std::make_shared<update_pos_pipe::data>(update_pos_pipe::data{data->holder,
        //       data->ctx}));
        //   }
        //   if (current_gid != -1) {
        //       update_proc_->input()->put_range(q);
        //   }
        // });
    }

    void start()
    {
        http_proc_->start();
    }

    void fini()
    {
        // stop processor in fini() because after stop we need to process the whole send queue
        http_proc_->stop();
    }

private:
    void send(
        ymod_webserver::http::stream_ptr stream,
        unsigned gid_from,
        unsigned gid_to,
        std::shared_ptr<yxiva::message> message)
    {
        auto holder = std::make_shared<stream_holder>(stream);
        holder->total_gids += gid_to - gid_from + 1;
        if (gid_from == gid_to)
        {
            list(holder, gid_from, message);
        }
        else
        {
            list_range(holder, gid_from, gid_to, message);
        }
    }

    void list(
        const std::shared_ptr<stream_holder>& holder,
        unsigned gid,
        std::shared_ptr<yxiva::message> message)
    {
        auto shards = shards_->get();
        auto shard = shard_from_gid(*shards, gid);
        if (!shard)
        {
            YLOG_CTX_GLOBAL(holder->stream->ctx(), error)
                << " error: failed to get shard by gid " << gid;
            return;
        }
        request_pq(*shard, "list", holder, gid, gid, message);
    }

    void list_range(
        const std::shared_ptr<stream_holder>& holder,
        unsigned gid_from,
        unsigned gid_to,
        std::shared_ptr<yxiva::message> message)
    {
        auto shards = shards_->get();
        for (auto& shard : *shards)
        {
            if (shard.end_gid >= gid_from && shard.start_gid <= gid_to)
            {
                request_pq(
                    shard,
                    "list_range",
                    holder,
                    std::max(shard.start_gid, gid_from),
                    std::min(shard.end_gid, gid_to),
                    message);
            }
        }
    }

    void request_pq(
        const shard_config::shard& shard,
        const string& request,
        const std::shared_ptr<stream_holder>& holder,
        unsigned gid_from,
        unsigned gid_to,
        const std::shared_ptr<yxiva::message>& message)
    {
        auto req = holder->stream->request();
        auto service = message->service;
        bool range = (gid_from != gid_to);
        auto args = boost::make_shared<ymod_pq::bind_array>();
        auto uid = message->uid;
        ymod_pq::push_const_string(args, service);
        ymod_pq::push_const_string(args, std::to_string(gid_from));
        if (range)
        {
            ymod_pq::push_const_string(args, std::to_string(gid_to));
        }
        auto subscriptions_handler = std::bind(
            &mod::handle_list, this, holder, uid, gid_from, message, std::placeholders::_1);
        auto pq_handler = boost::make_shared<pq_list_handler<decltype(subscriptions_handler)>>(
            service, list_chunk_size_, std::move(subscriptions_handler));
        auto fres =
            db_xtable->request(req->ctx(), shard.master.conninfo, request, args, pq_handler);
        fres.add_callback([fres, holder, pq_handler, gid_from]() {
            try
            {
                fres.get();
                pq_handler->flush();
            }
            catch (const std::exception& e)
            {
                YLOG_CTX_GLOBAL(holder->stream->ctx(), error) << gid_from << " error:" << e.what();
            }
        });
    }

    void handle_list(
        std::shared_ptr<stream_holder> holder,
        const string& uid,
        unsigned gid_from,
        std::shared_ptr<message> message,
        std::vector<sub_t> part)
    {
        auto ctx = holder->stream->ctx();
        YLOG_CTX_GLOBAL(ctx, info) << " chunk ready";
        try
        {
            holder->total += static_cast<unsigned>(part.size());
            auto queue = std::make_shared<std::vector<http_pipe::data_ptr>>();
            queue->reserve(part.size());
            for (auto& sub : part)
            {
                // ignore own mesh subscriptions
                if (uid == sub.uid)
                {
                    continue;
                }
                message->uid = sub.uid;
                if (!filter_accept(ctx, sub, *message))
                {
                    YLOG_CTX_GLOBAL(ctx, debug)
                        << sub.uid << " " << sub.service << " " << sub.id << " filtered";
                    continue;
                }
                queue->push_back(std::make_shared<http_pipe::data>(
                    http_pipe::data{ holder, ctx, std::move(sub), *message, false }));
            }
            http_proc_->input()->put_range(queue);
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_GLOBAL(ctx, error)
                << "handle_list gid_from=" << gid_from << " fail:" << e.what();
        }
    }

    bool filter_accept(task_context_ptr ctx, const sub_t& sub, message& message)
    {
        filter_set filter_set;
        filter::action action = filter::action::send_bright;
        if (auto parse_result = filter::parse(filter_set, sub.filter))
        {
            action = filter_set.apply(message);
        }
        else
        {
            YLOG_CTX_LOCAL(ctx, error)
                << "filter parse failed: " << parse_result.error_reason << " subid=" << sub.id;
        }
        if (action == filter::action::skip)
        {
            return false;
        }
        message.bright = (action == filter::action::send_bright);
        return true;
    }

    std::shared_ptr<ymod_pq::call> db_xtable;
    std::shared_ptr<yhttp::cluster_client> http_client;
    std::shared_ptr<http_pipe::proc> http_proc_;
    std::shared_ptr<yxiva::shard_config::storage> shards_;
    // std::shared_ptr<update_pos_pipe::proc> update_proc_;
    unsigned list_chunk_size_;
};

}}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::mesh::mod)
