#include <ymod_pq/call.h>
#include <yxiva/core/types.h>
#include <yxiva/core/subscriptions.h>
#include <yxiva/core/platforms.h>
#include <boost/lexical_cast.hpp>
#include <vector>

namespace yxiva { namespace mesh {

template <typename Callback>
class pq_list_handler : public ymod_pq::response_handler
{
    sub_t current_;
    std::vector<sub_t> result_;
    string service_;
    unsigned chunk_size_;
    Callback callback_;

public:
    enum
    {
        uid,
        id,
        callback,
        filter,
        extra_data,
        client,
        ttl,
        session_key,
        platform,

        column_cnt
    };

    pq_list_handler(const string& service, unsigned chunk_size, Callback&& callback)
        : service_(service), chunk_size_(chunk_size), callback_(std::move(callback))
    {
    }

    void handle_cell(unsigned /*row*/, unsigned col, const string& value, bool is_null)
    {
        switch (col)
        {
        case uid:
            current_.uid = value;
            break;
        case id:
            current_.id = value;
            break;
        case callback:
            current_.callback_url = value;
            break;
        case filter:
            current_.filter = value;
            break;
        case extra_data:
            current_.extra_data = value;
            break;
        case client:
            current_.client = value;
            break;
        case ttl:
            current_.ttl = is_null || value.empty() ? 0 : boost::lexical_cast<ttl_t>(value);
            break;
        case session_key:
            current_.session_key = value;
            break;
        case platform:
            current_.platform = platform::resolve_alias(value).name;
            break;
        }
    }

    void handle_row_end(unsigned /*row*/)
    {
        current_.service = service_;
        result_.push_back(current_);
        current_ = decltype(current_)();

        if (result_.size() >= chunk_size_)
        {
            decltype(result_) result_copy;
            result_copy.swap(result_);
            result_.reserve(result_copy.size());
            callback_(std::move(result_copy));
        }
    }

    unsigned column_count() const
    {
        return column_cnt;
    }

    void flush()
    {
        if (result_.size()) callback_(std::move(result_));
    }
};

class bulk_update_ack : public ymod_pq::response_handler
{
    local_id_t current_ = 0;
    std::vector<local_id_t> result_;

public:
    enum
    {
        local_id,
        column_cnt
    };

    void handle_cell(unsigned /*row*/, unsigned col, const string& value, bool is_null)
    {
        switch (col)
        {
        case local_id:
            current_ = is_null || value.empty() ? 0 : boost::lexical_cast<local_id_t>(value);
            break;
        }
    }

    void handle_row_end(unsigned /*row*/)
    {
        result_.push_back(current_);
        current_ = 0;
    }

    unsigned column_count() const
    {
        return column_cnt;
    }

    const decltype(result_)& get_data() const
    {
        return result_;
    }
};

}}
