#! /usr/bin/env python3

import datetime
import msgpack
import os
import OpenSSL.crypto
import re
from pprint import pprint


def juggler(results):
    resdict = {"CRIT": [], "WARN": [], "OK": []}

    for days, cert in results:
        if not days:
            # crit
            resdict["CRIT"].append((-999, cert))
        elif abs(days) < 14:
            resdict["CRIT"].append((days, cert))
        elif abs(days) <= 30:
            # warn
            resdict["WARN"].append((days, cert))
        else:
            # OK but show anyway
            resdict["OK"].append((days, cert))

    checkdesc = ""
    checkstatus = 0

    for status, desc in [(1, "WARN"), (2, "CRIT")]:
        if resdict[desc]:
            checkstatus = status
            checkdesc = desc + ": " + str(resdict[desc]).strip("[]") + " " + checkdesc
    print("{};{}".format(checkstatus, checkdesc))


def get_cert_validity(certfile_data):
    crypto = OpenSSL.crypto
    cert_re = re.compile(r"(-----BEGIN CERTIFICATE-----.*?-----END CERTIFICATE-----)", re.DOTALL)
    m = cert_re.search(certfile_data)
    try:
        cert = crypto.load_certificate(crypto.FILETYPE_PEM, m.group(0))
    except:
        return 999999999
    else:
        not_after = cert.get_notAfter()
        # print(not_after)
        not_after = datetime.date(int(not_after[:4]), int(not_after[4:6]), int(not_after[6:8]))
        today = datetime.date.today()
        valid_for = not_after - today
        return valid_for.days


def main():
    validities = []

    # populate dict from xconf dump
    xconf_dump_fname = "/var/xivamob/xconf/cached_config"
    f = open(xconf_dump_fname, "rb")
    secrets = msgpack.unpackb(f.read(), use_list=False)

    for key in secrets.keys():
        name = key[2]
        if name.startswith(b"apns:"):
            cert_data = secrets[key][0].decode("ascii", errors="ignore")
            cert_name = name[5:].decode("utf-8")
            validities.append((get_cert_validity(cert_data), cert_name))

    validities.sort()

    if "DEBUG" in os.environ:
        print(validities)
    else:
        juggler(validities)


if __name__ == "__main__":
    main()
