#pragma once

#include <yxiva_mobile/reports.h>
#include <yxiva_mobile/error.h>
#include <yxiva/core/json.h>
#include <yplatform/util/sstream.h>
#include <yplatform/util/safe_call.h>
#include <unordered_map>
#include <utility>

namespace yxiva { namespace mobile {

using yplatform::sstream;
using std::mutex;
using scoped_lock = std::unique_lock<mutex>;
namespace p = std::placeholders;

struct access_token
{
    bool expired() const
    {
        return std::time(nullptr) >= expires_at;
    }

    bool expires_soon() const
    {
        return std::time(nullptr) >= refresh_at;
    }

    error::code ec = error::code::success;
    string value;
    std::time_t refresh_at = 0;
    std::time_t expires_at = 0;
};

static const time_t RETRY_MAX = 86400; // 24h
static const std::vector<time_t> RETRY_MAP = { 1, 2, 4, 10, 20 };

template <typename GetTokenOperation>
class access_token_storage
    : public std::enable_shared_from_this<access_token_storage<GetTokenOperation>>
    , public boost::noncopyable
{
    using get_token_op = GetTokenOperation;
    using queue_handler_t = std::function<void(error::code, const string&)>;
    using queue_t = std::vector<queue_handler_t>;

public:
    access_token_storage(get_token_op get_token_op, string sid, string secret_key)
        : get_token_op_(std::move(get_token_op))
        , sid_(std::move(sid))
        , secret_key_(std::move(secret_key))
    {
    }

    // on get_token_op exception rolls state back and rethrows
    template <typename Handler>
    void async_get(Handler&& handler)
    {
        scoped_lock lock(guard_);

        if (sid_.empty() || secret_key_.empty())
        {
            lock.unlock();
            handler(error::code::invalid_cert, "");
            return;
        }

        if (data_.expired())
        {
            wait_queue_.push_back(std::forward<Handler>(handler));
            if (!updating_)
            {
                start_async_token_get(lock);
            }
            return;
        }

        if (data_.expires_soon() && !updating_)
        {
            start_async_token_get(lock);
        }

        auto data_copy = data_;
        if (lock) lock.unlock();
        handler(data_copy.ec, data_copy.value);
    }

    void reset(string sid, string secret_key)
    {
        scoped_lock lock(guard_);
        if (sid != sid_ || secret_key != secret_key_)
        {
            sid_ = sid;
            secret_key_ = secret_key;
            // Handle_data() calls reset if sid or key are changed.
            // An existing value is still available while updating.
            if (!updating_)
            {
                data_ = {};
            }
        }
    }

private:
    void start_async_token_get(scoped_lock& lock)
    {
        // SID and key are copied under lock
        auto sid_copy = sid_;
        auto secret_key_copy = secret_key_;
        // restored on exception
        updating_ = true;
        lock.unlock();
        try
        {
            get_token_op_(
                sid_copy,
                secret_key_copy,
                std::bind(
                    &access_token_storage::handle_data,
                    this->shared_from_this(),
                    p::_1,
                    sid_copy,
                    secret_key_copy));
        }
        catch (...)
        {
            lock.lock();
            updating_ = false;
            throw;
        }
    }

    // rethrows only its own exceptions
    void handle_data(const access_token& data, const string& sid, const string& secret_key)
    {
        scoped_lock lock(guard_);
        updating_ = false;

        queue_t queue;
        queue.swap(wait_queue_);

        try
        {
            // check for resets happened while performing the async call
            if (sid != sid_ || secret_key != secret_key_)
            {
                data_ = {};
            }
            else
            {
                data_ = data;
                // retry with no delay on internal errors and rare on unauthorized
                if (data_.ec == error::code::cloud_error)
                {
                    if (retry_map_pos_ + 1 < RETRY_MAP.size())
                    {
                        ++retry_map_pos_;
                    }
                    data_.expires_at = data_.refresh_at = RETRY_MAP[retry_map_pos_];
                }
                else if (data_.ec == error::code::invalid_cert)
                {
                    data_.expires_at = data_.refresh_at = RETRY_MAX;
                }
                else
                {
                    retry_map_pos_ = 0;
                }
            }
        }
        catch (...)
        {
            lock.unlock();
            call_handlers_safe(queue, error::code::internal_error, "");
            throw;
        }

        lock.unlock();
        call_handlers_safe(queue, data.ec, data.value);
    }

    void call_handlers_safe(const queue_t& queue, error::code ec, const string& token)
    {
        for (auto& handler : queue)
        {
            yplatform::safe_call(handler, ec, token);
        }
    }

    mutex guard_;
    get_token_op get_token_op_;
    string sid_;
    string secret_key_;
    access_token data_;
    std::vector<queue_handler_t> wait_queue_;
    bool updating_ = false;
    size_t retry_map_pos_ = 0;
};

}}
