#pragma once

#include <boost/system/error_code.hpp>
#include <vector>

namespace yxiva { namespace mobile {
namespace error {

enum code : int
{
    success = 0,
    invalid_token,
    no_cert,      // rename
    invalid_cert, // rename
    task_cancelled,
    cloud_error,
    invalid_payload,
    queue_overflow,
    shutdown,
    invalid_token_length,
    invalid_payload_length,
    data_compose_error,
    module_not_configured,
    cloud_not_available,
    internal_error,
    subscription_expired,
    invalid_subscription,
    partial_success,
    intermediate_success,
    device_rate_limit,
    network_error,
    not_implemented,

    COUNT
};

enum class condition
{
    success = 0,
    partial_success,
    bad_subscription,
    bad_request,
    internal_timeout,
    cloud_error,
    internal_error,
    cert_error,
    network_error
};

}

class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept(true) override
    {
        return "xivamob";
    }

    std::string message(int ev) const override
    {
        return error_names.at(ev);
    }

    static const error_category& instance()
    {
        static constexpr std::initializer_list<const char*> names = {
            "success",
            "invalid token",
            "no certificate",
            "invalid certificate",
            "task cancelled",
            "cloud error",
            "invalid payload",
            "queue overflow",
            "shutdown",
            "invalid token length",
            "invalid payload length",
            "data compose error",
            "module not configured properly",
            "cloud not available",
            "internal error",
            "subscription expired or unauthorized",
            "invalid subscription or endpoint",
            "partial success",
            "intermediate success",
            "device rate limit",
            "network error",
            "not implemented",
        };
        static_assert(
            names.size() == error::code::COUNT,
            "Error codes count doesn't correspond with error names count");
        static const error_category category(names);
        return category;
    }

private:
    error_category(std::initializer_list<const char*> names)
        : error_names(names.begin(), names.end())
    {
    }

    const std::vector<std::string> error_names;
};

inline boost::system::error_code make_error(error::code code = error::success)
{
    return boost::system::error_code(code, error_category::instance());
}

inline error::condition to_error_condition(error::code ec)
{
    switch (ec)
    {
    case error::success:
        return error::condition::success;

    case error::intermediate_success:
    case error::partial_success:
        return error::condition::partial_success;

    case error::invalid_token:
    case error::invalid_token_length:
    case error::subscription_expired:
    case error::invalid_subscription:
        return error::condition::bad_subscription;

    case error::invalid_payload:
    case error::invalid_payload_length:
    case error::data_compose_error:
    case error::device_rate_limit:
    case error::not_implemented:
        return error::condition::bad_request;

    case error::task_cancelled:
        return error::condition::internal_timeout;

    case error::cloud_error:
    case error::cloud_not_available:
        return error::condition::cloud_error;

    // Bad certificate cases should be separated from internal error cases
    // so that hub could keep the original error message
    case error::no_cert:
    case error::invalid_cert:
        return error::condition::cert_error;

    case error::shutdown:
    case error::queue_overflow:
    case error::module_not_configured:
    case error::internal_error:
        return error::condition::internal_error;

    case error::network_error:
        return error::condition::network_error;
    case error::COUNT:
        abort();
    }
    return error::condition::internal_error;
}

inline error::code calc_mixed_result(const std::vector<error::code>& chunk_results)
{
    bool has_success = false;
    bool has_error = false;

    auto result = error::data_compose_error;
    auto result_cond = error::condition::bad_request;

    for (auto& ec : chunk_results)
    {
        if (ec == error::invalid_cert)
        {
            return ec;
        }

        auto condition = to_error_condition(ec);
        switch (condition)
        {
        case error::condition::bad_subscription:
            return ec;
        case error::condition::success:
        case error::condition::partial_success:
            has_success = true;
            break;
        case error::condition::bad_request:
            has_error = true;
            break;
        case error::condition::internal_timeout:
            has_error = true;
            if (result_cond != condition && result_cond != error::condition::cloud_error)
            {
                result_cond = condition;
                result = ec;
            }
            break;
        case error::condition::cloud_error:
            has_error = true;
            if (result_cond == error::condition::bad_request)
            {
                result_cond = condition;
                result = ec;
            }
            break;
        case error::condition::cert_error:
        case error::condition::network_error:
        case error::condition::internal_error:
            has_error = true;
            if (result_cond != condition)
            {
                result_cond = condition;
                result = ec;
            }
        }
    }

    if (has_success) return has_error ? error::partial_success : error::success;
    return result;
}

}}

namespace boost { namespace system {

template <>
struct is_error_code_enum<yxiva::mobile::error_category>
{
    static const bool value = true;
};

}}
