#pragma once

#include <yxiva/core/types.h>
#include <yplatform/find.h>

// Logging and metrics collection.

namespace yxiva::mobile {

using yplatform::find_logger;

template <typename ContextPtr>
inline void report_new_request(ContextPtr ctx, const string& service, const string& transit_id)
{
    ctx->custom_log_data["service"] = service;
    ctx->custom_log_data["transit_id"] = transit_id;
}

template <typename ContextPtr>
inline void report_new_request_mobile(
    ContextPtr ctx,
    const string& service,
    const string& transit_id,
    const string& platform_app = {})
{
    report_new_request(ctx, service, transit_id);
    if (platform_app.size())
    {
        ctx->custom_log_data["app"] = platform_app;
    }
}

template <typename ContextPtr>
inline void report_check_result(ContextPtr ctx, const string& result)
{
    ctx->custom_log_data["check"] = result;
}

template <typename Context>
inline void report_response_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = {})
{
    LINFO_(ctx) << "ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_unsupported_platform(Context ctx, const string& platform)
{
    LWARN_(ctx) << "unsupported platform=" << platform;
}

template <typename Context>
inline void report_xconf_configuration_unpack_error(
    Context ctx,
    const string& conf_name,
    const string& error)
{
    LERR_(ctx) << "error on xconf config unpacking: name=" << conf_name << " error=" << error;
}

template <typename Context>
inline void report_apns_request_payload(Context ctx, const string& payload)
{
    auto logger = find_logger<yplatform::log::source>("dump");
    LINFO_(logger) << "apns request dump ctx=" << ctx->uniq_id() << " payload=" << payload;
}

template <typename Context>
inline void report_apns_payload_error(Context ctx, const string& error)
{
    LWARN_(ctx) << "apns payload error=" << error;
}

template <typename Context>
inline void report_fcm_request_payload(Context ctx, const string& payload)
{
    auto logger = find_logger<yplatform::log::source>("dump");
    LINFO_(logger) << "fcm request dump ctx=" << ctx->uniq_id() << " payload=" << payload;
}

template <typename Context>
inline void report_fcm_payload_error(Context ctx, const string& error)
{
    LWARN_(ctx) << "fcm payload error=" << error;
}

template <typename Context>
inline void report_apns_secret_load_failed(Context ctx, const string& error)
{
    LWARN_(ctx) << "apns secret load failed error=" << error;
}

template <typename Context>
inline void report_apns_sender_created(Context ctx, const char* when, const string& app)
{
    LINFO_(ctx) << "apns sender created [on " << when << "] app_name=" << app;
}

template <typename Context>
inline void report_apns_sender_destroyed(Context ctx, const string& app)
{
    LINFO_(ctx) << "apns sender destroyed app_name=" << app;
}

template <typename Context>
inline void report_apns_sender_create_failed(
    Context ctx,
    const char* when,
    const string& app,
    const char* exception)
{
    LERR_(ctx) << "apns sender create error [on " << when << "]: app_name=" << app
               << " exception=\"" << exception << "\"";
}

template <typename Context>
inline void report_apns_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "apns request failed: ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_apns_bearer_generated(
    Context ctx,
    const string& topic,
    const string& data,
    time_t valid_before)
{
    LINFO_(ctx) << "apns bearer generated for topic=" << topic << " data=" << data
                << " valid_before=" << valid_before;
}

template <typename Context>
inline void report_fcm_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "fcm request failed: ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_fcm_registration_id_changed(Context ctx)
{
    LINFO_(ctx) << "http response: error=\"registration_id changed\"";
}

template <typename Context>
inline void report_fcm_batch_response(Context ctx, int success, int failure, int canonical_ids)
{
    LINFO_(ctx) << "batch response: " << success << " tokens accepted, " << failure
                << " tokens with errors and " << canonical_ids << " new tokens available";
}

template <typename Context>
inline void report_fcm_api_key_found(Context ctx, const char* when, const string& app)
{
    LINFO_(ctx) << "fcm api key found on " << when << " app_name=" << app;
}

template <typename Context>
inline void report_mpns_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "mpns request failed: ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_wns_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "wns request failed: ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_wns_credentials_found(Context ctx, const char* when, const string& app)
{
    LINFO_(ctx) << "credentials found [on " << when << "]: app_name=" << app;
}

template <typename Context>
inline void report_wns_token_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "wns token request failed: ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_wns_invalid_payload(Context ctx, const string& error)
{
    LINFO_(ctx) << "wns xml conversion error=" << error;
}

template <typename Context>
inline void report_hms_token_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "hms token request failed: ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_hms_request_payload(Context ctx, const string& payload)
{
    auto logger = find_logger<yplatform::log::source>("dump");
    LINFO_(logger) << "hms request dump ctx=" << ctx->uniq_id() << " payload=" << payload;
}

template <typename Context>
inline void report_hms_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "hms request failed: ec=" << ec.message() << " error=" << error_extra;
}

template <typename Context>
inline void report_hms_credentials_found(Context ctx, const char* when, const string& app)
{
    LINFO_(ctx) << "credentials found [on " << when << "]: app_name=" << app;
}

template <typename Context>
inline void report_hms_payload_error(Context ctx, const string& error)
{
    LWARN_(ctx) << "hms payload error=" << error;
}

template <typename Context>
inline void report_webpush_keys_generated(Context ctx)
{
    LINFO_(ctx) << "webpush keys generated";
}

template <typename Context>
inline void report_webpush_keys_generation_falied(Context ctx, const string& error)
{
    LERR_(ctx) << "webpush keys generation failed: " << error;
}

template <typename Context>
inline void report_vapid_keys_ready(Context ctx, const string& public_b64)
{
    LINFO_(ctx) << "vapid keys ready, public key is " << public_b64;
}

template <typename Context>
inline void report_vapid_keys_load_falied(Context ctx, const string& error)
{
    LERR_(ctx) << "vapid keys load failed error=" << error;
}

template <typename Context>
inline void report_vapid_generated(
    Context ctx,
    const string& origin,
    const string& vapid,
    const time_duration& d)
{
    LINFO_(ctx) << "vapid generated for origin=" << origin << " vapid=" << vapid
                << " tm=" << yplatform::time_traits::to_string(d) << "s";
}

template <typename Context>
inline void report_vapid_generation_failed(Context ctx, const string& origin, const string& error)
{
    LINFO_(ctx) << "vapid generation failed for origin=" << origin << " error=" << error;
}

template <typename Context>
inline void report_invalid_webpush_endpoint(Context ctx, const string& error)
{
    LWARN_(ctx) << "webpush endpoint parse error=" << error;
}

template <typename Context>
inline void report_invalid_webpush_subscription(Context ctx, const string& error)
{
    LWARN_(ctx) << "webpush subscription parse error=" << error;
}

template <typename Context>
inline void report_no_webpush_encryption_key(Context ctx)
{
    LERR_(ctx) << "no encryption key is available";
}

template <typename Context>
inline void report_webpush_request_error(
    Context ctx,
    const boost::system::error_code& ec,
    const string& error_extra = string())
{
    LINFO_(ctx) << "webpush request failed: ec=" << ec.message() << " error=" << error_extra;
}

}