#pragma once

#include "sender_p8.h"
#include "sender_pem.h"
#include "settings.h"
#include <chrono>
#include <random>

namespace yxiva::mobile::apns {

template <typename Clock, typename Hash>
class migration_sender_impl : public sender
{
public:
    migration_sender_impl(
        sender_ptr sender_src,
        sender_ptr sender_dst,
        time_t updated_at,
        duration duration,
        const yplatform::log::source& parent_logger)
        : sender(parent_logger)
        , sender_src_(std::move(sender_src))
        , sender_dst_(std::move(sender_dst))
        , updated_at_(updated_at)
        , duration_(duration_cast<seconds>(duration).count())
    {
    }

    void push(const mobile_task_context_ptr& ctx, callback_t&& cb) override
    {
        get_sender(ctx->uniq_id())->push(ctx, std::move(cb));
    }

    const string& secret_type() const override
    {
        static const string secret_type = "migration";
        return secret_type;
    }

    yplatform::ptree get_stats() const override
    {
        yplatform::ptree stats;
        append_stats(stats, sender_src_);
        append_stats(stats, sender_dst_);
        return stats;
    }

private:
    sender_ptr sender_src_;
    sender_ptr sender_dst_;
    time_t updated_at_;
    int64_t duration_;

    sender_ptr get_sender(const string& unique_id)
    {
        remove_sender_src_when_migration_finished();

        if (!sender_src_)
        {
            return sender_dst_;
        }

        return randomly_choose_sender(unique_id);
    }

    sender_ptr randomly_choose_sender(const string& unique_id)
    {
        auto elapsed = Clock::to_time_t(Clock::now()) - updated_at_;
        int chance = duration_ ? round(100.0 * elapsed / duration_) : 100;
        chance = std::max(0, std::min(100, chance));

        int uniform_random_value = Hash()(unique_id) % 101;

        return uniform_random_value < chance ? sender_dst_ : sender_src_;
    }

    void remove_sender_src_when_migration_finished()
    {
        if (sender_src_ && Clock::to_time_t(Clock::now()) >= updated_at_ + duration_)
        {
            sender_src_.reset();
        }
    }

    void append_stats(yplatform::ptree& stats, const sender_ptr& sender) const
    {
        if (!sender)
        {
            return;
        }

        for (auto& [key, value] : sender->get_stats())
        {
            stats.put(key, value.data());
        }
    }
};

using migration_sender = migration_sender_impl<std::chrono::system_clock, std::hash<string>>;

}
