#include "mod.h"
#include <yxiva_mobile/reports.h>
#include <boost/algorithm/string/predicate.hpp>

namespace yxiva::mobile::apns {

mod::mod(yplatform::reactor& reactor) : reactor_(yplatform::reactor::make_not_owning_copy(reactor))
{
}

void mod::init(const yplatform::ptree& data)
{
    settings_ = std::make_shared<apns_settings>();
    settings_->load(data);
    senders_.logger(logger());
    senders_.start(read_secrets(), *settings(), reactor_);
}

void mod::reload(const yplatform::ptree& data)
{
    auto st = std::make_shared<apns_settings>();
    st->load(data);
    settings(st);
}

yplatform::ptree mod::get_stats() const
{
    yplatform::ptree result;
    result.put_child("pool", senders_.get_stats());
    return result;
}

void mod::push(mobile_task_context_ptr ctx, callback_t&& cb)
{
    auto MAX_PAYLOAD_SIZE = settings()->max_payload_size;

    if (std::find_if_not(ctx->token.begin(), ctx->token.end(), isxdigit) != ctx->token.end())
    {
        cb(make_error(error::invalid_token));
        return;
    }
    if (ctx->payload.size() > MAX_PAYLOAD_SIZE)
    {
        cb(make_error(error::invalid_payload_length));
        return;
    }
    auto result = prepare_payload(ctx);
    if (!result)
    {
        report_apns_payload_error(ctx, result.error_reason);
        cb(make_error(error::invalid_payload));
        return;
    }
    if (ctx->payload.size() > MAX_PAYLOAD_SIZE)
    {
        cb(make_error(error::invalid_payload_length));
        return;
    }
    if (settings()->dump_payload)
    {
        report_apns_request_payload(ctx, ctx->payload);
    }
    // TODO use h2 timeout from ctx
    if (auto sender = senders_.find_sender(ctx->app_name))
    {
        sender->push(ctx, std::move(cb));
    }
    else
    {
        cb(make_error(error::no_cert));
    }
}

void mod::check(mobile_task_context_ptr /*ctx*/, callback_t&& cb)
{
    cb(make_error(error::not_implemented));
}

void mod::update_application(const application_config& app)
{
    senders_pool::secrets_map secrets;
    secrets[app.app_name].actual = { app.secret_key, app.environment };
    secrets[app.app_name].backup = { app.key_backup, app.environment };
    secrets[app.app_name].updated_at = app.updated_at;
    senders_.update(secrets, *settings());
}

operation::result mod::prepare_payload(mobile_task_context_ptr& ctx)
{
    auto& url = ctx->request->url;
    auto icollapse_id = url.param_find("x-collapse-id");
    if (icollapse_id != url.params.end())
    {
        if (icollapse_id->second.size() > settings()->max_collapse_id_size)
        {
            return "collapse-id constraint violated";
        }
        ctx->headers.emplace_back("apns-collapse-id", icollapse_id->second);
        ctx->request->ctx()->custom_log_data["apns-collapse-id"] = icollapse_id->second;
    }

    json_value payload_js;
    auto error = payload_js.parse(ctx->payload);
    if (error || !payload_js.is_object()) return "payload must be a valid json object";

    if (payload_js.has_member("aps"))
        // valid in case of hardcoded mail service repacking in xivahub
        return operation::success;

    auto icustom_aps = url.param_find("x-aps");
    if (icustom_aps != url.params.end())
    {
        // aps.content-available, if needed, must be present inside x-aps
        json_value aps_js;
        auto error = aps_js.parse(icustom_aps->second);
        if (error || !aps_js.is_object())
        {
            return "bad x-aps parameter";
        }
        payload_js["aps"] = std::move(aps_js);
    }
    else
    {
        // if no aps set and custom key provided - set aps.content-available
        if (payload_js.size())
        {
            payload_js["aps"]["content-available"] = 1;
        }
    }

    if (payload_js.empty()) return "empty payload";

    auto push_type = detect_push_type(ctx, payload_js);
    ctx->headers.emplace_back("apns-push-type", push_type);
    if (push_type == "background")
    {
        ctx->headers.emplace_back("apns-priority", "5"); // required value for type=background
    }
    else
    {
        auto ipriority = url.param_find("x-priority");
        if (ipriority != url.params.end())
        { // the default value is 10
            ctx->headers.emplace_back("apns-priority", ipriority->second);
        }
    }
    ctx->payload = payload_js.stringify();
    return operation::success;
}

string mod::detect_push_type(const mobile_task_context_ptr& ctx, const json_value& payload)
{
    if (boost::algorithm::ends_with(ctx->app_name, ".voip"))
    {
        return "voip";
    }
    else if (payload.has_member("aps") && payload["aps"].has_member("alert"))
    {
        // Assume silent notifications don't have aps.alert.
        return "alert";
    }
    else
    {
        return "background";
    }
}

senders_pool::secrets_map mod::read_secrets()
{
    auto st = settings();
    senders_pool::secrets_map secret_buffs;
    for (auto secret : st->secret_files)
    {
        auto& name = secret.first;

        auto& filename_actual = secret.second.second;
        if (filename_actual.size())
        {
            secret_buffs[name].actual = { read_secret_from_file(filename_actual),
                                          app_environment::automatic };
        }

        auto& filename_backup = secret.second.first;
        if (filename_backup.size())
        {
            secret_buffs[name].backup = { read_secret_from_file(filename_backup),
                                          app_environment::automatic };
        }

        secret_buffs[name].updated_at = std::time(nullptr);
    }
    return secret_buffs;
}

string mod::read_secret_from_file(const string& filename)
{
    std::stringstream secret_stream;
    std::fstream secret_file(filename, std::ifstream::in);
    if (!secret_file)
    {
        report_apns_secret_load_failed(
            logger(), "can't open file with secret: filename=\"" + filename + "\"");
    }
    secret_stream << secret_file.rdbuf();
    return secret_stream.str();
}

apns_settings_ptr mod::settings()
{
    read_lock lock(guard_);
    return settings_;
}

void mod::settings(apns_settings_ptr st)
{
    write_lock lock(guard_);
    settings_ = st;
}

}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::mobile::apns::mod)
