#pragma once

#include "settings.h"
#include <yxiva_mobile/push_task_context.h>
#include <yxiva_mobile/reports.h>
#include <yxiva/core/platforms.h>
#include <yxiva/core/ec_crypto.h>
#include <yxiva/core/types.h>
#include <yplatform/encoding/base64.h>
#include <chrono>
#include <random>

namespace yxiva::mobile::apns {

using namespace yxiva::apns;

template <typename Clock>
class p8_bearer_impl
{
public:
    p8_bearer_impl(const p8_token& token, const apns_settings& settings)
        : token_(token)
        , lifetime_(duration_cast<seconds>(settings.p8_lifetime).count())
        , valid_before_(0)
    {
    }

    const string& get(const mobile_task_context_ptr& ctx)
    {
        if (Clock::to_time_t(Clock::now()) >= valid_before_)
        {
            auto data = update();
            update_valid_before();
            report_apns_bearer_generated(ctx, token_.topic, data, valid_before_);
        }

        return bearer_;
    }

private:
    string update()
    {
        yplatform::json_value header_json;
        header_json["alg"] = "ES256";
        header_json["kid"] = token_.key_id;
        auto header = header_json.stringify();

        yplatform::json_value claim_json;
        claim_json["iss"] = token_.issuer_key;
        claim_json["iat"] = Clock::to_time_t(Clock::now());
        auto claim = claim_json.stringify();

        string data;
        data += yplatform::base64_urlsafe_encode(header.begin(), header.end());
        data += ".";
        data += yplatform::base64_urlsafe_encode(claim.begin(), claim.end());

        auto priv_key = yxiva::ec_crypto::read_pem_buf(token_.key);
        auto sign = yxiva::ec_crypto::sign(data.data(), data.size(), priv_key);
        bearer_ = data;
        bearer_ += ".";
        bearer_ += yplatform::base64_urlsafe_encode(sign.begin(), sign.end());

        return data;
    }

    void update_valid_before()
    {
        valid_before_ = Clock::to_time_t(Clock::now()) + lifetime_;
    }

    const p8_token& token_;
    time_t lifetime_;
    time_t valid_before_;
    string bearer_;
};

using p8_bearer = p8_bearer_impl<std::chrono::system_clock>;

}
