#include "sender.h"
#include <yxiva_mobile/error.h>
#include <yxiva_mobile/reports.h>

namespace yxiva::mobile::apns {

namespace http_error = ymod_httpclient::http_error;

const string APNS_REASON_BADDEVICETOKEN = "BadDeviceToken";

sender::sender(const yplatform::log::source& parent_logger)
{
    logger(parent_logger);
}

yplatform::ptree sender::get_stats() const
{
    yplatform::ptree stats;
    stats.put("requests_sent_" + secret_type() + "_cumulative", requests_sent_);
    return stats;
}

boost::system::error_code sender::handle_http_error(
    mobile_task_context_ptr ctx,
    const boost::system::error_code& err)
{
    report_apns_request_error(ctx, err);

    error::code result;
    switch (err.value())
    {
    case http_error::ssl_error:
        result = error::invalid_cert;
        break;
    case http_error::task_canceled:
    case http_error::request_timeout:
        result = error::task_cancelled;
        break;
    default:
        result = error::internal_error;
    }
    return make_error(result);
}

boost::system::error_code sender::handle_http_status(
    mobile_task_context_ptr ctx,
    yhttp::response resp,
    string& result_body)
{
    if (resp.status / 100 == 2)
    {
        json_value response_fields;
        auto message_id_it = resp.headers.find("apns-id");
        if (message_id_it != resp.headers.end())
        {
            response_fields["apns_message_id"] = message_id_it->second;
        }
        result_body = response_fields.stringify();
        return make_error();
    }

    report_apns_request_error(ctx, {}, std::to_string(resp.status) + " " + resp.body);

    json_value response_js;
    auto error = response_js.parse(resp.body);
    if (!error)
    {
        result_body = response_js["reason"].stringify();
    }

    error::code result = error::cloud_error;
    if (resp.status / 100 == 5)
    {
        result = error::cloud_error;
    }
    else if (
        resp.status == 410 || (resp.status == 400 && result_body == APNS_REASON_BADDEVICETOKEN))
    {
        result = error::invalid_token;
    }
    else if (resp.status == 400 || resp.status == 405)
    {
        result = error::data_compose_error;
    }
    else if (resp.status == 403)
    {
        result = error::invalid_cert;
    }
    else if (resp.status == 413)
    {
        result = error::invalid_payload_length;
    }
    else if (resp.status == 429)
    {
        result = error::device_rate_limit;
    }

    return make_error(result);
}

const string& sender::topic_from_url_params(
    const mobile_task_context_ptr& ctx,
    const string& default_topic) const
{
    auto& url = ctx->request->url;
    auto itopic = url.param_find("x-topic");
    return itopic != url.params.end() ? itopic->second : default_topic;
}

}
