#include "sender_p8.h"
#include <yxiva/core/platforms.h>

namespace yxiva::mobile::apns {

using namespace yxiva::apns;

sender_p8::sender_p8(
    boost::asio::io_service& io,
    const sender::secret& secret,
    const apns_settings& settings,
    const yplatform::log::source& parent_logger)
    : sender(parent_logger)
    , io_(io)
    , http2_settings_(settings.http2)
    , token_(secret.first)
    , bearer_(token_, settings)
{
    url_ = resolve_certificate_type(token_.type) == certificate_type::production ?
        settings.url :
        settings.sandbox_url;
    reconnect();
}

void sender_p8::push(const mobile_task_context_ptr& ctx, callback_t&& cb)
{
    ymod_httpclient::h2::headers headers;
    headers.reserve(ctx->headers.size() + 3);
    headers.assign(ctx->headers.begin(), ctx->headers.end());
    headers.emplace_back("apns-topic", topic_from_url_params(ctx, token_.topic));
    headers.emplace_back(
        "apns-expiration", std::to_string(ctx->ttl ? std::time(nullptr) + ctx->ttl : 0));
    headers.emplace_back("authorization", "bearer " + bearer_.get(ctx));

    auto request =
        ymod_httpclient::h2::request::POST(url_ + ctx->token, std::move(headers), ctx->payload);
    httpclient_->async_run(
        ctx,
        std::move(request),
        [this, self = shared_from_this(), ctx, cb = std::move(cb)](
            const boost::system::error_code& err, yhttp::response resp) {
            ++requests_sent_;
            string result_body;
            cb(err ? handle_http_error(ctx, err) : handle_http_status_p8(ctx, resp, result_body),
               result_body);
        });
}

const string& sender_p8::secret_type() const
{
    static const string secret_type = "p8";
    return secret_type;
}

void sender_p8::reconnect()
{
    httpclient_ = std::make_shared<ymod_httpclient::h2::client>(io_, http2_settings_);
    httpclient_->logger(logger());
}

boost::system::error_code sender_p8::handle_http_status_p8(
    mobile_task_context_ptr ctx,
    yhttp::response resp,
    string& result_body)
{
    json_value response_js;
    string reason;
    auto error = response_js.parse(resp.body);
    if (!error) reason = response_js["reason"].stringify();

    if (resp.status == 429 && reason == "TooManyProviderTokenUpdates")
    {
        reconnect();
    }

    return handle_http_status(ctx, resp, result_body);
}

}
