#include "senders_pool.h"
#include <yxiva_mobile/reports.h>

namespace yxiva::mobile::apns {

using namespace yxiva::apns;

sender_ptr make_secret_specific_sender(
    boost::asio::io_service& io,
    const sender::secret& secret,
    const apns_settings& settings,
    const yplatform::log::source& parent_logger)
{
    if (pem_certificate::contains_pem(secret.first))
    {
        return std::make_shared<sender_pem>(io, secret, settings, parent_logger);
    }
    else
    {
        return std::make_shared<sender_p8>(io, secret, settings, parent_logger);
    }
}

void senders_pool::start(
    secrets_map secrets,
    const apns_settings& settings,
    yplatform::reactor_ptr reactor)
{
    reactor_ = reactor;
    pool_ = std::make_shared<senders_map>();
    known_secrets_ = std::make_shared<secrets_map>();
    update(secrets, settings, "start");
}

void senders_pool::update(
    const secrets_map& new_secrets,
    const apns_settings& settings,
    const char* when)
{
    auto curr_pool = pool();
    auto new_pool = std::make_shared<senders_map>(*curr_pool);
    auto curr_known_secrets = known_secrets();
    auto updated_known_secrets = std::make_shared<secrets_map>(*curr_known_secrets);
    for (auto& [secret_name, secret] : new_secrets)
    {
        auto& secret_buff = secret.actual.first;

        auto iknown_secret = updated_known_secrets->find(secret_name);
        bool secret_changed = (iknown_secret == updated_known_secrets->end()) ||
            (iknown_secret->second.actual != secret.actual);

        if (secret_changed) (*updated_known_secrets)[secret_name] = secret;

        if (secret_buff.empty() || secret_changed) remove_sender(new_pool, secret_name);

        if (not secret_buff.empty() && secret_changed)
        {
            if (auto sender = create_sender(secret_name, secret, settings, when))
                new_pool->insert(std::make_pair(secret_name, sender));
        }
    }
    pool(new_pool);
    known_secrets(updated_known_secrets);
}

sender_ptr senders_pool::find_sender(const string& app_name)
{
    auto senders = pool();
    auto isender = senders->find(app_name);
    if (isender != senders->end())
    {
        return isender->second;
    }
    return nullptr;
}

yplatform::ptree senders_pool::get_stats() const
{
    yplatform::ptree stats;
    auto senders = pool();
    for (auto& sender_pair : *senders)
    {
        auto& name = sender_pair.first;
        auto& sender = sender_pair.second;
        stats.add_child({ "app-" + name, '/' }, sender->get_stats());
    }
    return stats;
}

senders_pool::senders_map_ptr senders_pool::pool() const
{
    read_lock lock(guard_);
    return pool_;
}

void senders_pool::pool(senders_map_ptr updated)
{
    write_lock lock(guard_);
    pool_ = updated;
}

senders_pool::secrets_map_ptr senders_pool::known_secrets() const
{
    read_lock lock(guard_);
    return known_secrets_;
}

void senders_pool::known_secrets(secrets_map_ptr updated)
{
    write_lock lock(guard_);
    known_secrets_ = updated;
}

void senders_pool::remove_sender(senders_map_ptr pool, const string& secret_name)
{
    auto isender = pool->find(secret_name);
    if (isender != pool->end())
    {
        pool->erase(isender);
        report_apns_sender_destroyed(logger(), secret_name);
    }
}

sender_ptr senders_pool::create_sender(
    const string& secret_name,
    const secret& secret,
    const apns_settings& settings,
    const char* when)
{
    try
    {
        auto sender =
            make_secret_specific_sender(*reactor_->io(), secret.actual, settings, logger());
        handle_pem_to_p8_migration(sender, secret, settings);

        report_apns_sender_created(logger(), when, secret_name);
        return sender;
    }
    catch (const std::exception& ex)
    {
        report_apns_sender_create_failed(logger(), when, secret_name, ex.what());
    }
    return nullptr;
}

void senders_pool::handle_pem_to_p8_migration(
    sender_ptr& sender,
    const secret& secret,
    const apns_settings& settings)
{
    auto& duration = settings.p8_activation_duration;
    if (sender->secret_type() == "p8" && pem_certificate::contains_pem(secret.backup.first) &&
        pem_valid_during_migration(secret.backup.first, secret.updated_at, duration))
    {
        auto sender_pem =
            make_secret_specific_sender(*reactor_->io(), secret.backup, settings, logger());

        sender = std::make_shared<migration_sender>(
            std::move(sender_pem), std::move(sender), secret.updated_at, duration, logger());
    }
}

bool senders_pool::pem_valid_during_migration(
    const string& cert,
    time_t updated_at,
    duration duration)
{
    try
    {
        pem_certificate::validate(cert);
        auto exp = pem_certificate::get_expiration_time(cert);
        return updated_at + duration_cast<seconds>(duration).count() < exp;
    }
    catch (...)
    {
        return false;
    }
}

}
