#pragma once

#include <yxiva/core/types.h>
#include <ymod_httpclient/h2/settings.h>
#include <yplatform/net/settings.h>
#include <yplatform/time_traits.h>
#include <yplatform/net/ssl_settings.h>
#include <boost/filesystem.hpp>
#include <map>
#include <string>
#include <exception>

namespace yxiva { namespace mobile { namespace apns {

using duration = yplatform::time_traits::duration;

struct apns_settings
{
    std::map<string, std::pair<string, string>> secret_files;
    ymod_httpclient::h2::settings http2;
    size_t max_payload_size = 4096;
    size_t max_collapse_id_size = 64;
    string url;
    string sandbox_url;
    duration p8_lifetime;
    duration p8_lifetime_gap;
    duration p8_activation_duration;
    bool dump_payload = false;

    void load(const yplatform::ptree& data)
    {
        if (auto path = data.get<string>("secrets", ""); path != "")
        {
            load_secrets(data.get<string>("secrets"));
        }
        http2.parse_ptree(data.get_child("http2"));
        max_payload_size = data.get("max_payload_size", max_payload_size);
        url = data.get<string>("url");
        sandbox_url = data.get<string>("sandbox_url");
        p8_lifetime = data.get<duration>("p8_lifetime");
        p8_lifetime_gap = data.get<duration>("p8_lifetime_gap");
        p8_activation_duration = data.get<duration>("p8_activation_duration");
        dump_payload = data.get("dump_payload", dump_payload);
    }

private:
    void load_secrets(const string& secrets_path)
    {
        namespace fs = boost::filesystem;
        if (!fs::exists(secrets_path))
        {
            throw std::runtime_error("secrets not loaded from FS, no such folder " + secrets_path);
        }
        fs::directory_iterator end_it; // default construction yields past-the-end
        for (fs::directory_iterator it(secrets_path); it != end_it; ++it)
        {
            auto extension =
                fs::is_regular_file(it->status()) ? it->path().extension().string() : "";
            if (extension == ".pem" || extension == ".p8" || extension == ".backup")
            {
                string app_name = it->path().stem().string();
                std::replace(app_name.begin(), app_name.end(), '_', '.');
                auto& secret_path = secret_files[app_name];
                if (extension == ".backup")
                {
                    secret_path.first = boost::filesystem::absolute(it->path()).string();
                }
                else
                {
                    secret_path.second = boost::filesystem::absolute(it->path()).string();
                }
            }
        }
    }
};

typedef std::shared_ptr<apns_settings> apns_settings_ptr;

}}}
