#pragma once

#include "request.h"
#include "settings.h"
#include <yxiva_mobile/push_task_context.h>
#include <yxiva_mobile/ipusher.h>
#include <yxiva_mobile/error.h>
#include <yxiva_mobile/mutex.h>
#include <yxiva_mobile/reports.h>
#include <yxiva/core/platforms.h>
#include <yxiva/core/http_handlers.h>
#include <yxiva/core/packing.hpp>
#include <ymod_httpclient/call.h>
#include <ymod_httpclient/response_handler.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <boost/algorithm/string/replace.hpp>
#include <functional>
#include <map>
#include <memory>
#include <set>

namespace yxiva { namespace mobile { namespace fcm {

class http_pusher
    : public batch_pusher
    , public yplatform::module
{
public:
    void init(const yplatform::ptree& conf)
    {
        auto st = std::make_shared<http_pusher_settings>();
        st->load(conf);
        settings_ = st;
        http_client_ = yplatform::find<yhttp::call, std::shared_ptr>("http_client");
    }

    void reload(const yplatform::ptree& data)
    {
        auto st = std::make_shared<http_pusher_settings>();
        st->load(data);

        write_lock lock(guard_);
        st->api_keys = settings_->api_keys;
        settings_ = st;
    }

    void start()
    {
        auto st = settings();
        if (st->host_push.empty())
        {
            throw std::runtime_error("FCM push host is not configured");
        }
        if (st->host_check.empty())
        {
            throw std::runtime_error("FCM check host is not configured");
        }
    }

    void stop()
    {
    }

    void push(mobile_task_context_ptr task, callback_t&& cb) override
    {
        if (!is_usable_token(task->token, platform::FCM))
        {
            respond(cb, make_error(error::invalid_token));
            return;
        }

        auto api_key = find_api_key(task->app_name);
        if (!api_key)
        {
            respond(cb, make_error(error::no_cert));
            return;
        }

        do_push<fcm_push_request>(std::move(task), *api_key, std::move(cb));
    }

    void check(mobile_task_context_ptr task, callback_t&& cb) override
    {
        auto api_key = find_api_key(task->app_name);
        if (!api_key)
        {
            respond(cb, make_error(error::no_cert));
            return;
        }

        do_check<fcm_check_request>(std::move(task), *api_key, std::move(cb));
    }

    void update_application(const application_config& app) override
    {
        {
            write_lock lock(guard_);
            auto new_settings = std::make_shared<http_pusher_settings>(*settings_);
            if (app.secret_key.empty())
            {
                new_settings->api_keys.erase(app.app_name);
            }
            else
            {
                auto credential = yxiva::fcm::parse_secret_key(app.secret_key);
                new_settings->api_keys[app.app_name] = credential.api_key;
            }
            settings_ = new_settings;
        }
        report_fcm_api_key_found(logger(), "update", app.app_name);
    }

    void push_batch(batch_task_context_ptr task, result_callback_t&& cb) override
    {
        auto api_key = find_api_key(task->app_name);
        if (!api_key)
        {
            respond(cb, make_error(error::no_cert));
            return;
        }

        do_push<fcm_push_batch_request>(std::move(task), *api_key, std::move(cb));
    }

private:
    template <typename Request, typename Task, typename Callback>
    void do_push(Task&& task, const string& api_key, Callback&& cb)
    {
        yhttp::request http_request;
        auto prepared = prepare_push_request(task, api_key, http_request);
        if (!prepared.first)
        {
            report_fcm_payload_error(task, prepared.first.error_reason);
            respond(cb, make_error(prepared.second));
            return;
        }

        if (settings()->dump_payload)
        {
            report_fcm_request_payload(task, *http_request.body);
        }

        make_call(
            std::make_shared<Request>(std::move(task), std::move(http_request), std::move(cb)));
    }

    template <typename Request, typename Task, typename Callback>
    void do_check(Task&& task, const string& api_key, Callback&& cb)
    {
        auto http_request = prepare_check_request(task, api_key);
        make_call(
            std::make_shared<Request>(std::move(task), std::move(http_request), std::move(cb)));
    }

    std::optional<string> find_api_key(const string& app_name) const
    {
        auto st = settings();
        auto iapi_key = st->api_keys.find(app_name);
        if (iapi_key == st->api_keys.end()) return {};
        return iapi_key->second;
    }

    template <typename Task>
    std::pair<operation::result, error::code> prepare_push_request(
        const Task& task,
        const string& api_key,
        yhttp::request& http_request)
    {
        static const std::vector<request_param> supported_params = {
            { "x-notification", json_type::tobject, FCM_MAX_NOTIFICATION_LENGTH },
            { "x-restricted_package_name", json_type::tstring },
            { "x-priority", json_type::tstring },
            { "x-collapse_key", json_type::tstring },
            { "x-delay_while_idle", json_type::tbool }
        };

        json_value js_body;

        if (task->payload.size())
        {
            if (task->payload.size() > FCM_MAX_DATA_LENGTH)
                return { "payload too large", error::invalid_payload_length };
            json_value data;
            auto error = data.parse(task->payload);
            if (error || !data.is_object())
            {
                return { "payload must be a valid json object", error::invalid_payload };
            }
            js_body["data"] = std::move(data);
        }

        for (auto& param : supported_params)
        {
            auto raw = task->request->url.param_value(param.name, "");
            if (raw.empty()) continue;
            if (raw.size() > param.max_length)
                return { "too large " + param.name + " parameter", error::invalid_payload_length };
            json_value value;
            if (auto error = value.parse(raw); error || value.type() != param.type)
            {
                return { "bad " + param.name + " parameter", error::invalid_payload };
            }
            js_body[param.name.substr(2) /*skip "x-"*/] = std::move(value);
        }

        if (js_body["data"].empty() && js_body["notification"].empty())
        {
            return { "empty payload", error::invalid_payload };
        }

        fill_recipient(task, js_body);
        js_body["time_to_live"] = task->ttl;
        // js_body["restricted_package_name"] = task->app_name;

        string headers;
        headers.reserve(128);
        headers.append("Content-Type: application/json\r\n")
            .append("Authorization: key=")
            .append(api_key)
            .append("\r\n");
        string body = js_body.stringify();

        auto st = settings();
        http_request =
            yhttp::request::POST(st->host_push + st->url_push, std::move(headers), std::move(body));

        return { operation::success, error::success };
    }

    yhttp::request prepare_check_request(const mobile_task_context_ptr& task, const string& api_key)
    {
        string headers;
        headers.reserve(128);
        headers.append("Authorization: key=").append(api_key).append("\r\n");

        auto st = settings();
        string url = st->host_check + st->url_check + "/" + task->token;

        return yhttp::request::GET(std::move(url), std::move(headers));
    }

    void fill_recipient(const mobile_task_context_ptr& task, json_value& data)
    {
        data["to"] = task->token;
    }

    void fill_recipient(const batch_task_context_ptr& task, json_value& data)
    {
        data["registration_ids"] = std::move(task->tokens);
    }

    template <typename Request>
    void make_call(Request req)
    {
        http_client_->async_run(
            req->task,
            std::move(req->http_request),
            settings()->http_opts,
            [this, req](auto ec, auto response) {
                if (ec)
                {
                    report_fcm_request_error(
                        req->task, ec, std::to_string(response.status) + " " + response.body);
                    this->respond(
                        req->cb,
                        make_error(
                            (ec == yhttp::errc::request_timeout ||
                             ec == yhttp::errc::task_canceled) ?
                                error::task_cancelled :
                                error::internal_error));
                    return;
                }
                this->handle_response(response, req);
            });
    }

    template <typename Request>
    void handle_response(const yhttp::response& response, Request req)
    {
        if (response.status / 100 != 2)
        {
            string reason = boost::replace_all_copy(response.body, "\n", "\\n");
            report_fcm_request_error(req->task, {}, reason);
            handle_http_error(response.status, reason, req);
            return;
        }

        json_value parsed_body;
        if (auto error = parsed_body.parse(response.body))
        {
            report_fcm_request_error(req->task, {}, "invalid json: \"" + *error + "\"");
            respond(req->cb, make_error(error::cloud_error));
            return;
        }

        handle_json_response(parsed_body, req);
        // @todo: log stats if everything is ok
    }

    void handle_response(const yhttp::response& response, fcm_check_request_ptr req)
    {
        auto& token = req->task->token;
        auto code = response.status;
        string error;
        if (json_value parsed_body; json_parse(parsed_body, response.body))
        {
            error = parsed_body["error"].to_string("");
        }
        else
        {
            error = boost::replace_all_copy(response.body, "\n", "\\n");
        }

        auto result = make_check_result(token, code, error);

        report_check_result(req->task->request->ctx(), result["code"].to_string(""));
        respond(req->cb, make_error(error::success), result.stringify());
    }

    json_value make_check_result(const string& token, int code, const string& error)
    {
        json_value result;
        result["token"] = token;
        result["code"] = std::to_string(code);
        result["error"] = error;
        return result;
    }

    template <typename Request>
    void handle_http_error(unsigned code, const string& response, Request req)
    {
        if (code == 400 and
            response.substr(0, strlen("INVALID_REGISTRATION")) == "INVALID_REGISTRATION")
        {
            respond(req->cb, make_error(error::invalid_token));
        }
        else
        {
            if (code == 401)
            {
                respond(req->cb, make_error(error::invalid_cert));
            }
            else
            {
                respond(req->cb, make_error(error::cloud_error));
            }
        }
    }

    void handle_json_response(const json_value& response, fcm_push_request_ptr req)
    {
        string error, new_id, new_reg_id;

        // Response containing "results" array, with only result having message_id or error,
        // and possibly new registration_id, corresponds to request with a single push token.
        if (response.has_member("results") && response["results"].is_array() &&
            !response["results"].empty())
        {
            auto&& result = response["results"][0UL];
            error = result["error"].to_string("");
            new_id = result["message_id"].to_string("");
            new_reg_id = result["registration_id"].to_string("");
            // Response without "results" array and containing "message_id" or "error"
            // on root level, corresponds to request with a single topic
        }
        else if (response.has_member("error") || response.has_member("message_id"))
        {
            error = response["error"].to_string("");
            new_id = response["message_id"].to_string("");
        }
        else
        {
            report_fcm_request_error(req->task, {}, "wrong json response format");
            respond(req->cb, make_error(error::cloud_error), "invalid fcm response");
            return;
        }

        json_value special_fields;
        if (error.size())
        {
            report_fcm_request_error(req->task, {}, "http response: error=\"" + error + "\"");
            special_fields["error"] = error;
        }
        if (new_id.size())
        {
            special_fields["message_id"] = new_id;
        }
        if (new_reg_id.size())
        {
            report_fcm_registration_id_changed(req->task);
            special_fields["new_token"] = new_reg_id;
        }
        respond(req->cb, error_code_from_message(error), special_fields.stringify());
    }

    void handle_json_response(const json_value& response, fcm_push_batch_request_ptr req)
    {
        int success = response["success"].get<int>(-1);
        int failure = response["failure"].get<int>(-1);
        int canonical_ids = response["canonical_ids"].get<int>(-1);
        auto fcm_results = response["results"];
        if (success == -1 || failure == -1 || canonical_ids == -1 || fcm_results.is_null())
        {
            report_fcm_request_error(req->task, {}, "wrong json response format");
            respond(req->cb, make_error(error::cloud_error));
            return;
        }
        report_fcm_batch_response(req->task, success, failure, canonical_ids);

        for (auto&& res : fcm_results.array_items())
        {
            json_value special_fields;
            auto error = res["error"].to_string("");
            auto new_id = res["message_id"].to_string("");
            auto new_reg_id = res["registration_id"].to_string("");
            if (error.size())
            {
                special_fields["error"] = error;
            }
            if (new_id.size())
            {
                special_fields["message_id"] = new_id;
            }
            if (new_reg_id.size())
            {
                special_fields["new_token"] = new_reg_id;
            }
            req->cb(error_code_from_message(error), std::move(special_fields), false);
        }
        respond(req->cb, make_error(error::success));
    }

    boost::system::error_code error_code_from_message(const string& error_message)
    {
        if (token_errors.count(error_message)) return make_error(error::invalid_token);
        else if (error_message.size())
            return make_error(error::cloud_error);
        return make_error(error::success);
    }

    void respond(
        callback_t& cb,
        const boost::system::error_code& err,
        const string& body = string())
    {
        cb(err, body);
    }

    void respond(result_callback_t& cb, const boost::system::error_code& err)
    {
        cb(err, json_value{}, true);
    }

    std::shared_ptr<http_pusher_settings> settings() const
    {
        read_lock lock(guard_);
        return settings_;
    }

private:
    static const std::set<string> token_errors;
    mutable shared_mutex guard_;
    std::shared_ptr<http_pusher_settings> settings_;
    std::shared_ptr<yhttp::call> http_client_;
};

}}}
