#pragma once

#include <yxiva_mobile/reports.h>
#include <yxiva_mobile/error.h>
#include <yxiva_mobile/access_token_storage.h>
#include <yxiva/core/json.h>
#include <ymod_httpclient/call.h>
#include <yplatform/encoding/url_encode.h>
#include <yplatform/util/safe_call.h>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <unordered_map>

namespace yxiva { namespace mobile {

struct async_get_hms_access_token_op
{
    using handler_t = std::function<void(const access_token&)>;

    yplatform::task_context_ptr ctx{ boost::make_shared<yplatform::task_context>() };
    string auth_url;
    yplatform::time_traits::duration timeout;
    std::shared_ptr<yhttp::call> httpclient;
    string name;

    void operator()(const string& client_id, const string secret_key, const handler_t& handler)
    {
        auto body = yhttp::form_encode({ { "grant_type", "client_credentials" },
                                         { "client_id", client_id },
                                         { "client_secret", secret_key } });
        auto request = yhttp::request::POST(auth_url, std::move(body));
        auto options = yhttp::options();
        options.timeouts.total = timeout;
        httpclient->async_run(
            ctx,
            std::move(request),
            std::bind(
                &async_get_hms_access_token_op::handle_response, this, p::_1, p::_2, handler));
    }

    void handle_response(
        const boost::system::error_code& ec,
        yhttp::response response,
        const handler_t& handler)
    {
        access_token result;
        try
        {
            if (!ec && response.status == 200)
            {
                result = parse_json_response(response);
                return handler(result);
            }

            if (ec)
            {
                result.ec = error::code::internal_error;
            }
            else
            {
                result.ec = error::code::cloud_error;
            }
        }
        catch (const std::exception& e)
        {
            result.ec = error::code::internal_error;
            report_hms_token_request_error(ctx, make_error(result.ec), e.what());
            handler(result);
            return;
        }
        report_hms_token_request_error(
            ctx, ec, std::to_string(response.status) + " " + response.body);
        handler(result);
    }

    access_token parse_json_response(const yhttp::response& response)
    {
        static const time_t MIN_EXPIRES_IN = 300U; // 5 minutes

        access_token result;
        json_value body_json;
        if (auto error = body_json.parse(response.body))
        {
            report_hms_token_request_error(ctx, {}, *error);
            result.ec = error::code::cloud_error;
            return result;
        }
        auto expires_in = body_json["expires_in"].get<uint64_t>(0);
        result.value = body_json["access_token"].to_string_view(true);
        boost::trim(result.value);
        boost::replace_all(result.value, "\\", "");
        if (expires_in < MIN_EXPIRES_IN || result.value.empty())
        {
            report_hms_token_request_error(ctx, {}, body_json.stringify());
            result.ec = error::code::cloud_error;
        }
        else
        {
            result.expires_at = std::time(nullptr) + expires_in;
            result.refresh_at = result.expires_at - MIN_EXPIRES_IN;
        }
        return result;
    }
};

}}
