#pragma once

#include <yxiva_mobile/error.h>
#include <yxiva_mobile/push_task_context.h>

namespace yxiva { namespace mobile {

inline void fill_default_click_action(const mobile_task_context_ptr& ctx, json_value& body)
{
    auto android_notification = body["message"]["android"]["notification"];

    auto click_action = android_notification["click_action"];
    if (!click_action.has_member("type"))
    {
        click_action["type"] = 1;
    }
    if (!click_action.has_member("intent"))
    {
        click_action["intent"] = ctx->transit_id.size() ? ctx->transit_id : ctx->uniq_id();
    }
}

inline void fill_default_title_body(json_value& body)
{
    auto message = body["message"];
    auto android_notification = message["android"]["notification"];
    if (android_notification.has_member("title") && android_notification.has_member("body"))
    {
        message["notification"]["title"].assign_copy(android_notification["title"]);
        message["notification"]["body"].assign_copy(android_notification["body"]);
    }
    else
    {
        throw std::runtime_error("notification message must include title and body");
    }
}

inline std::pair<operation::result, error::code> fill_default_notification_fields(
    const mobile_task_context_ptr& ctx,
    json_value& body)
{
    try
    {
        fill_default_click_action(ctx, body);
        fill_default_title_body(body);
    }
    catch (const std::exception& ex)
    {
        return { ex.what(), error::code::invalid_payload };
    }
    return { operation::success, error::code::success };
}

inline std::pair<operation::result, error::code> prepare_hms_message(
    const mobile_task_context_ptr& ctx,
    json_value& body)
{
    static const std::vector<string> supported_params{ "x-notification",
                                                       "x-collapse_key",
                                                       "x-urgency" };
    json_value message_data;
    auto error = message_data.parse(ctx->payload);
    if (error || !message_data.is_object())
    {
        return { "payload must be a valid json object", error::invalid_payload };
    }
    body["message"]["data"] = message_data.stringify();
    body["message"]["token"].push_back(ctx->token);
    body["message"]["android"]["ttl"] = std::to_string(ctx->ttl) + "s";

    for (const auto& param : supported_params)
    {
        auto param_value = ctx->request->url.param_value(param, "");
        if (param_value.empty()) continue;
        json_value param_value_json;
        if (auto error = param_value_json.parse(param_value))
        {
            return { "bad " + param + " parameter", error::invalid_payload };
        }
        body["message"]["android"][param.substr(2) /*skip "x-"*/] = param_value_json;
    }

    if (ctx->request->url.param_count("x-notification"))
    {
        return fill_default_notification_fields(ctx, body);
    }
    return { operation::success, error::code::success };
}

}}