#pragma once

#include "access_tokens.h"
#include "message.h"
#include <yxiva_mobile/error.h>
#include <yxiva_mobile/ipusher.h>
#include <yxiva_mobile/reports.h>
#include <yxiva/core/platforms.h>
#include <ymod_httpclient/call.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/spinlock.h>
#include <vector>

namespace yxiva { namespace mobile {

namespace p = std::placeholders;

#define HMS_MAX_DATA_LENGTH 4 * 1024 // 4 Kb

struct hms_settings
{
    struct app_data
    {
        string name;
        string client_id;
        string key;
    };

    string auth_url;
    string push_url;
    yplatform::time_traits::duration auth_timeout;
    yplatform::time_traits::duration send_timeout;
    std::vector<app_data> test_apps;
    bool dump_payload = false;

    void load(const yplatform::ptree& conf)
    {
        auth_url = conf.get<string>("auth_url");
        push_url = conf.get<string>("push_url");
        auth_timeout = conf.get<yplatform::time_traits::duration>("auth_timeout");
        send_timeout = conf.get<yplatform::time_traits::duration>("send_timeout");
        yplatform::ptree empty;
        for (auto& app : conf.get_child("test_apps", empty))
        {
            if (app.first == "app")
                test_apps.push_back(app_data{ app.second.get<string>("name"),
                                              app.second.get<string>("client_id"),
                                              app.second.get<string>("key") });
        }
        dump_payload = conf.get("dump_payload", dump_payload);
    }
};

class mod_hms
    : public ipusher
    , public yplatform::module
{
    using spinlock = yplatform::spinlock;
    using scoped_lock = std::unique_lock<spinlock>;
    using access_token_storage_t = access_token_storage<async_get_hms_access_token_op>;

    struct request_data
    {
        mobile_task_context_ptr ctx;
        size_t position = 0;
        string access_token;
        callback_t callback;
    };

public:
    void init(const yplatform::ptree& conf)
    {
        settings_.load(conf);
        httpclient_ = yplatform::find<yhttp::call, std::shared_ptr>("http_client");
        for (auto& app : settings_.test_apps)
        {
            add_app_token(app.name, app.client_id, app.key);
        }
    }

    void push(mobile_task_context_ptr ctx, callback_t&& cb) override
    {
        if (ctx->payload.size() > HMS_MAX_DATA_LENGTH)
        {
            respond(cb, make_error(error::invalid_payload));
            return;
        }

        scoped_lock lock(tokens_guard_);
        if (!app_exists(ctx->app_name))
        {
            lock.unlock();
            return respond(cb, make_error(error::no_cert));
        }
        auto token_storage = app_tokens_[ctx->app_name];
        auto client_id = client_ids_[ctx->app_name];
        lock.unlock();

        token_storage->async_get(std::bind(
            &mod_hms::handle_get_access_token,
            shared_from(this),
            p::_1,
            p::_2,
            ctx,
            client_id,
            std::move(cb)));
    }

    void check(mobile_task_context_ptr /*ctx*/, callback_t&& cb) override
    {
        cb(make_error(error::not_implemented));
    }

    void update_application(const application_config& conf) override
    {
        auto credential = yxiva::hms::parse_secret_key(conf.secret_key);

        scoped_lock lock(tokens_guard_);
        // erase app only if both client_id and secret_key are empty
        if (conf.secret_key.empty())
        {
            app_tokens_.erase(conf.app_name);
            client_ids_.erase(conf.app_name);
        }
        else
        {
            add_app_token(conf.app_name, credential.id, credential.secret);
        }

        lock.unlock();
        report_hms_credentials_found(logger(), "update", conf.app_name);
    }

private:
    void handle_get_access_token(
        error::code ec,
        const string& access_token,
        mobile_task_context_ptr ctx,
        const string& client_id,
        const callback_t& cb)
    {
        if (ctx->is_cancelled()) return;

        if (ec)
        {
            cb(make_error(ec));
            return;
        }

        do_send(ctx, client_id, access_token, cb);
    }

    void do_send(
        mobile_task_context_ptr ctx,
        const string& client_id,
        const string& access_token,
        const callback_t& cb)
    {
        string url = settings_.push_url + "/v1/" + client_id + "/messages:send";

        string headers = "Authorization: Bearer " + access_token + "\r\n";

        json_value body;
        auto prepared = prepare_hms_message(ctx, body);
        if (!prepared.first)
        {
            report_hms_payload_error(ctx, prepared.first.error_reason);
            respond(cb, make_error(prepared.second));
            return;
        }

        if (settings_.dump_payload)
        {
            report_hms_request_payload(ctx, body.stringify());
        }

        yhttp::options opts;
        opts.timeouts.total = settings_.send_timeout;
        httpclient_->async_run(
            ctx,
            yhttp::request::POST(url, headers, body.stringify()),
            opts,
            std::bind(&mod_hms::handle_sent, shared_from(this), p::_1, ctx, p::_2, cb));
    }

    void handle_sent(
        const boost::system::error_code& ec,
        mobile_task_context_ptr ctx,
        yhttp::response response,
        const callback_t& cb)
    {
        if (ctx->is_cancelled()) return;

        if (ec)
        {
            report_hms_request_error(ctx, ec);
            respond(cb, make_error(error::network_error));
            return;
        }

        auto result_code = http_status_to_error_code(response.status);
        if (result_code)
        {
            report_hms_request_error(
                ctx, ec, std::to_string(response.status) + " " + response.body);
        }

        json_value parsed_body;
        if (auto error = parsed_body.parse(response.body))
        {
            report_hms_request_error(ctx, ec, "invalid json \"" + *error + "\"");
            respond(cb, make_error(error::cloud_error));
            return;
        }

        json_value special_fields;
        extract_response_fields(result_code, parsed_body, special_fields);

        respond(cb, make_error(result_code), special_fields.stringify());
    }

    error::code http_status_to_error_code(int status)
    {
        error::code result_code;

        if (status / 100 == 2)
        {
            result_code = error::success;
        }
        else if (status == 401 || status == 403)
        {
            result_code = error::invalid_cert;

            // not sure about 400 - no way to detect it's a bad URI or bad request
        }
        else if (status == 400 || status == 404 || status == 410)
        {
            result_code = error::invalid_subscription;
        }
        else if (status == 413)
        {
            result_code = error::invalid_payload_length;
        }
        else
        {
            result_code = error::cloud_error;
        }
        return result_code;
    }

    void extract_response_fields(
        error::code result_code,
        const json_value& body,
        json_value& out_fields)
    {
        string code, msg, request_id;
        if (result_code)
        {
            code = body["code"].to_string("");
            msg = body["msg"].to_string("");
        }
        request_id = body["requestId"].to_string("");

        if (code.size()) out_fields["code"] = code;
        if (msg.size()) out_fields["msg"] = msg;
        if (request_id.size()) out_fields["requestId"] = request_id;
    }

    bool app_exists(const string& app_name) const
    {
        return app_tokens_.count(app_name) && client_ids_.count(app_name);
    }

    void add_app_token(const string& name, const string& client_id, const string& key)
    {
        auto& token_storage = app_tokens_[name];
        if (!token_storage)
        {
            async_get_hms_access_token_op get_token_op;
            get_token_op.auth_url = settings_.auth_url;
            get_token_op.timeout = settings_.auth_timeout;
            get_token_op.httpclient = httpclient_;
            get_token_op.name = name;
            token_storage = std::make_shared<access_token_storage_t>(get_token_op, client_id, key);
        }
        else
        {
            token_storage->reset(client_id, key);
        }
        client_ids_[name] = client_id;
    }

    void respond(
        const callback_t& cb,
        const boost::system::error_code& err,
        const string& body = string())
    {
        cb(err, body);
    }

    hms_settings settings_;
    spinlock tokens_guard_;
    std::shared_ptr<yhttp::call> httpclient_;
    std::unordered_map<string, std::shared_ptr<access_token_storage_t>> app_tokens_;
    std::unordered_map<string, string> client_ids_;
};

}}
