#include "convert.h"
#include <yxiva_mobile/reports.h>
#include <yxiva/core/json.h>
#include <yxiva/core/quote_xml.h>
#include <yplatform/util/sstream.h>
#include <exception>

namespace yxiva::mobile {
namespace {

using formatters::quoteXML;
using ct = std::char_traits<char>;
using sstream = yplatform::sstream;

const string XML_CLEAR = R"( Action="Clear"></)";
const string XML_ID = " Id=\"";
const string XML_TEMPLATE = " Template=\"";
const string XML_NOTIFICATION_OPEN = R"(<wp:Notification xmlns:wp="WPNotification">)";
const string XML_NOTIFICATION_CLOSE = R"(</wp:Notification>)";
const string XML_NS = "wp:";
const char* MPNS_XML_TAG_TILE = "Tile";
const char* MPNS_XML_TAG_TOAST = "Toast";

void write_clear_element(sstream& xml, const string& name)
{
    xml << "<" << XML_NS << quoteXML(name) << XML_CLEAR << XML_NS << quoteXML(name) << ">";
}

void write_clear_block(sstream& xml, const json_value& root)
{
    if (root.is_array())
    {
        for (auto&& val : root.array_items())
        {
            auto element = val.to_string("");
            if (!element.empty())
            {
                write_clear_element(xml, element);
            }
        }
    }
    else
    {
        auto element = root.to_string("");
        if (!element.empty())
        {
            write_clear_element(xml, element);
        }
    }
}

operation::result convert_default(
    string& xml_string,
    const string& data,
    wns_notification_type::type type)
{
    static const string tag_text = "Text1";
    static const string::size_type toast_base_len = 10 + XML_NOTIFICATION_OPEN.size() +
        XML_NOTIFICATION_CLOSE.size() + 4 * XML_NS.size() + 2 * ct::length(MPNS_XML_TAG_TOAST) +
        2 * tag_text.size();

    switch (type)
    {
    case wns_notification_type::toast:
    {
        sstream(xml_string, toast_base_len + data.size())
            << XML_NOTIFICATION_OPEN << "<" << XML_NS << MPNS_XML_TAG_TOAST << "><" << XML_NS
            << tag_text << ">" << quoteXML(data) << "</" << XML_NS << tag_text << "></" << XML_NS
            << MPNS_XML_TAG_TOAST << ">" << XML_NOTIFICATION_CLOSE;
        break;
    }
    default:
        return "default conversion not supported for this notification type";
    }
    return operation::success;
}

operation::result tag_name_from_type(wns_notification_type::type type, string& tag)
{
    switch (type)
    {
    case wns_notification_type::tile:
        tag = MPNS_XML_TAG_TILE;
        break;
    case wns_notification_type::toast:
        tag = MPNS_XML_TAG_TOAST;
        break;
    default:
        return "notification type not supported";
    }
    return operation::success;
}
}

operation::result convert_mpns(
    const string& input,
    wns_notification_type::type notification_type,
    string& output)
{
    // TODO return bad request for type::badge

    // pass raw payload as is
    if (notification_type == wns_notification_type::raw)
    {
        output = input;
        return operation::success;
    }
    json_value root;
    auto error = root.parse(input);
    // if can't parse json, construct default xml for given
    // push type and input string
    if (error || root.is_string() || root.is_number())
    {
        return convert_default(output, input, notification_type);
    }
    if (!root.is_object())
    {
        return "invalid json: need object to convert to xml";
    }
    try
    {
        string tag_name;
        auto tag_res = tag_name_from_type(notification_type, tag_name);
        if (!tag_res) return tag_res;
        yplatform::sstream xml(output, 2 * input.size());
        // open notification tag and push type tag
        xml << XML_NOTIFICATION_OPEN << "<" << XML_NS << tag_name;
        // write id and template attrs and terminate tag
        auto id_val = root.cref["id"].to_string("");
        auto template_val = root.cref["template"].to_string("");
        root.remove_member("id");
        root.remove_member("template");
        if (!id_val.empty()) xml << XML_ID << quoteXML(id_val) << "\"";
        if (!id_val.empty()) xml << XML_TEMPLATE << quoteXML(template_val) << "\"";
        xml << ">";
        // write tags with Action=Clear
        write_clear_block(xml, root.cref["clear"]);
        root.remove_member("clear");
        // write all other child nodes
        for (auto it = root.members_begin(); it != root.members_end(); ++it)
        {
            auto name = it.key();
            auto field_val = (*it).to_string("");
            xml << "<" << XML_NS << quoteXML(name) << ">" << quoteXML(field_val) << "</" << XML_NS
                << quoteXML(name) << ">";
        }
        // close all tags
        xml << "</" << XML_NS << tag_name << ">" << XML_NOTIFICATION_CLOSE;
        return operation::success;
    }
    catch (const std::exception& e)
    {
        return e.what();
    }
}

}
