#pragma once

#include "convert.h"
#include "mod_wns/chunks.h"
#include <yxiva_mobile/reports.h>
#include <yxiva_mobile/ipusher.h>
#include <ymod_httpclient/call.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/util/sstream.h>
#include <memory>

namespace yxiva { namespace mobile {

using yplatform::sstream;
namespace p = std::placeholders;

struct mpns_settings
{
    yplatform::time_traits::duration send_timeout;

    void load(const yplatform::ptree& conf)
    {
        send_timeout = conf.get<yplatform::time_traits::duration>("send_timeout");
    }
};

class mod_mpns
    : public ipusher
    , public yplatform::module
{
    struct request_data
    {
        mobile_task_context_ptr ctx;
        size_t position = 0;
        std::vector<chunk> chunks;
        std::vector<error::code> chunk_results;
        callback_t callback;
    };

public:
    void init(const yplatform::ptree& conf)
    {
        settings_.load(conf);
        httpclient_ = yplatform::find<yhttp::call, std::shared_ptr>("http_client");
    }

    void push(mobile_task_context_ptr ctx, callback_t&& cb) override
    {
        auto chunked_data =
            make_chunked_data<request_data>(ctx->request->url, ctx->payload, convert_mpns);

        chunked_data->ctx = ctx;
        chunked_data->callback = std::move(cb);

        process_chunks(chunked_data);
    }

    void check(mobile_task_context_ptr /*ctx*/, callback_t&& cb) override
    {
        cb(make_error(error::not_implemented));
    }

    void update_application(const application_config& /*app*/) override
    {
        // nothing to do
    }

private:
    void process_chunks(std::shared_ptr<request_data> chunked_data)
    {
        if (chunked_data->position < chunked_data->chunks.size())
        {
            if (is_partial_success(chunked_data))
            {
                chunked_data->callback(make_error(error::intermediate_success));
            }
            send(chunked_data);
        }
        else
        {
            chunked_data->callback(make_error(calc_mixed_result(chunked_data->chunk_results)));
        }
    }

    bool is_partial_success(std::shared_ptr<request_data>& chunked_data)
    {
        bool result = false;
        for (auto& ec : chunked_data->chunk_results)
        {
            if (ec == error::success)
            {
                result = true;
                break;
            }
        }
        return result;
    }

    void send(std::shared_ptr<request_data>& chunked_data)
    {
        assert(chunked_data->position < chunked_data->chunks.size());

        static const string XHEADERS[wns_notification_type::COUNT] = {
            "X-WindowsPhone-Target: toast\r\n", "", "X-WindowsPhone-Target: token\r\n", ""
        };
        static const string COMMON_HEADERS = "Content-Type: text/xml\r\n";

        auto& chunk = chunked_data->chunks[chunked_data->position];
        string headers;
        sstream(headers, 100) << XHEADERS[chunk.type] << COMMON_HEADERS;

        yhttp::options opts;
        opts.timeouts.total = settings_.send_timeout;
        httpclient_->async_run(
            chunked_data->ctx,
            yhttp::request::POST(chunked_data->ctx->token, headers, string(chunk.payload)),
            opts,
            std::bind(&mod_mpns::handle_sent, shared_from(this), p::_1, p::_2, chunked_data));
    }

    void handle_sent(
        const boost::system::error_code& ec,
        yhttp::response response,
        std::shared_ptr<request_data>& chunked_data)
    {
        auto& ctx = chunked_data->ctx;

        if (ctx->is_cancelled()) return;

        auto result_code = ec ? error::network_error : http_status_to_error_code(response.status);

        if (result_code)
        {
            report_mpns_request_error(
                ctx, ec, std::to_string(response.status) + " " + response.body);
            chunked_data->callback(make_error(result_code));
            return;
        }

        // TODO log?
        /*
          X-DeviceConnectionStatus: disconnected
          X-NotificationStatus: Received/Suppressed/QueueFull
          X-SubscriptionStatus: Active
          X-MessageID
        */

        chunked_data->chunk_results.push_back(result_code);
        chunked_data->position++;
        process_chunks(chunked_data);
    }

    error::code http_status_to_error_code(int status)
    {
        error::code result_code;

        if (status / 100 == 2)
        {
            result_code = error::success;
        }
        else if (status == 401)
        {
            result_code = error::invalid_cert;

            // not sure about 400 - no way to detect it's a bad URI or bad request
        }
        else if (status == 400 || status == 404 || status == 410)
        {
            result_code = error::invalid_subscription;
        }
        else
        {
            result_code = error::cloud_error;
        }
        return result_code;
    }

    mpns_settings settings_;
    std::shared_ptr<yhttp::call> httpclient_;
};

}}
