#include "impl.h"

#include <yxiva_mobile/push_task_context.h>
#include <yxiva_mobile/error.h>
#include <yxiva_mobile/find_xconf.h>
#include <yxiva/core/methods/check.h> // isalnum_ext())
#include <ymod_webserver/server.h>
#include <ymod_webserver/methods/transform.h>
#include <ymod_webserver/expirable_stream.h>
#include <yplatform/find.h>
#include <yplatform/time_traits.h>

namespace yxiva::mobile::web {
namespace {
static const string LOAD_LIMIT = "*load-limit*";
}

void ping(ymod_webserver::http::stream_ptr stream)
{
    stream->result(ymod_webserver::codes::ok, "pong");
}

void impl::init(const yplatform::ptree& data)
{
    request_stats_ = std::make_shared<request_stats>();
    request_stats_webpush_ = std::make_shared<request_stats>();
    settings_ = make_shared<settings>();
    settings_->load(data);

    using ymod_webserver::transformer;
    using ymod_webserver::argument;
    using ymod_webserver::optional_argument;
    using ymod_webserver::validator;
    auto push_transformer = transformer(
        argument<string>("app", validator(isalnum_ext)),
        argument<string>("token"),
        argument<string>("payload"),
        optional_argument<unsigned>("ttl", settings_->max_ttl),
        optional_argument<string>("service", ""),
        optional_argument<string>("transit_id", ""));
    auto windows_push_transformer = transformer(
        argument<string>("app", validator(isalnum_ext)),
        argument<string>("token"),
        optional_argument<string>("payload", ""),
        optional_argument<unsigned>("ttl", settings_->max_ttl),
        optional_argument<string>("service", ""),
        optional_argument<string>("transit_id", ""));
    auto webpush_transformer = transformer(
        argument<string>("subscription"),
        optional_argument<string>("payload", ""),
        optional_argument<unsigned>("ttl", settings_->max_ttl),
        optional_argument<string>("service", ""),
        optional_argument<string>("transit_id", ""));
    auto batch_transformer = transformer(
        argument<string>("app", validator(isalnum_ext)),
        argument<string>("payload"),
        argument<string>("tokens"),
        optional_argument<unsigned>("ttl", settings_->max_ttl),
        optional_argument<string>("service", ""),
        optional_argument<string>("transit_id", ""));
    auto check_transformer = transformer(
        argument<string>("app", validator(isalnum_ext)),
        argument<string>("token"),
        argument<string>("service"));
    auto webserver_module = yplatform::find<ymod_webserver::server>("web_server");
    fcm_proxy = std::make_shared<mobile_proxy>(
        "fcm_http_pusher", settings_->get_timeout("fcm"), settings_->max_ttl);
    fcm_batch_proxy = std::make_shared<batch_proxy>(
        "fcm_http_pusher",
        settings_->get_timeout("fcm"),
        settings_->max_ttl); // TODO: fcm batch timeout?
    apns_proxy = std::make_shared<mobile_proxy>(
        "apns_pusher", settings_->get_timeout("apns"), settings_->max_ttl);
    wns_proxy = std::make_shared<mobile_proxy>(
        "wns_pusher", settings_->get_timeout("wns"), settings_->max_ttl);
    mpns_proxy = std::make_shared<mobile_proxy>(
        "mpns_pusher", settings_->get_timeout("mpns"), settings_->max_ttl);
    hms_proxy = std::make_shared<mobile_proxy>(
        "hms_pusher", settings_->get_timeout("hms"), settings_->max_ttl);
    webpush_proxy =
        std::make_shared<::yxiva::mobile::web::webpush_proxy>(settings_->get_timeout("webpush"));
    fcm_proxy->logger(logger());
    fcm_batch_proxy->logger(logger());
    apns_proxy->logger(logger());
    wns_proxy->logger(logger());
    mpns_proxy->logger(logger());
    hms_proxy->logger(logger());
    webpush_proxy->logger(logger());

    webserver_module->bind("", { "/ping" }, ping);
    webserver_module->bind(
        "", { "/push/gcm", "/push/fcm" }, *fcm_proxy, push_transformer); // gcm_compatibility
    webserver_module->bind("", { "/check/fcm" }, *fcm_proxy, check_transformer);
    webserver_module->bind(
        "",
        { "/batch_push/gcm", "/batch_push/fcm" },
        *fcm_batch_proxy,
        batch_transformer); // gcm_compatibility
    webserver_module->bind("", { "/push/apns" }, *apns_proxy, push_transformer);
    webserver_module->bind("", { "/push/wns" }, *wns_proxy, windows_push_transformer);
    webserver_module->bind("", { "/push/mpns" }, *mpns_proxy, windows_push_transformer);
    webserver_module->bind("", { "/push/hms" }, *hms_proxy, windows_push_transformer);
    webserver_module->bind("", { "/webpush" }, *webpush_proxy, webpush_transformer);

    webserver_module->bind("", { LOAD_LIMIT }, [](ymod_webserver::http::stream_ptr stream) {
        stream->result(ymod_webserver::codes::service_unavailable, "rate limit");
    });
    webserver_module->set_custom_key_extractor(
        "",
        [request_stats_ = request_stats_,
         request_stats_webpush_ = request_stats_webpush_,
         settings_ = settings_](ymod_webserver::http::stream_ptr stream) {
            if (stream->request()->url.path.size() && stream->request()->url.path[0] == "webpush")
            {
                auto current_rps = request_stats_webpush_->new_request();
                if (current_rps > static_cast<int>(settings_->rps_limit_webpush))
                {
                    return LOAD_LIMIT;
                }
                request_stats_webpush_->accept_request();
            }
            else
            {
                auto current_rps = request_stats_->new_request();
                if (stream->request()->url.path.size() &&
                    stream->request()->url.path[0] != "ping" &&
                    current_rps > static_cast<int>(settings_->rps_limit))
                {
                    return LOAD_LIMIT;
                }
                request_stats_->accept_request();
            }
            return stream->request()->url.make_full_path();
        });

    // subscribe xconf and redirect new configurations to proper pusher
    std::weak_ptr<yplatform::module> wptr = this->shared_from_this();
    find_xconf()->subscribe_updates(
        ymod_xconf::config_type::MOBILE, [wptr, this](ymod_xconf::conf_list_ptr confs) {
            auto ptr = wptr.lock();
            if (!ptr) return;
            for (auto& conf : confs->items)
            {
                application_config app;
                try
                {
                    unpack(conf.configuration, app);
                    app.platform = platform::resolve_alias(app.platform).name; // gcm_compatibility
                    if (app.platform == platform::FCM)
                    {
                        fcm_proxy->update_application(app);
                    }
                    else if (app.platform == platform::APNS)
                    {
                        apns_proxy->update_application(app);
                    }
                    else if (app.platform == platform::WNS)
                    {
                        wns_proxy->update_application(app);
                    }
                    else if (app.platform == platform::MPNS)
                    {
                        mpns_proxy->update_application(app);
                    }
                    else if (app.platform == platform::HMS)
                    {
                        hms_proxy->update_application(app);
                    }
                    else
                    {
                        report_unsupported_platform(logger(), app.platform);
                    }
                }
                catch (const std::exception& ex)
                {
                    report_xconf_configuration_unpack_error(logger(), conf.name, ex.what());
                }
            }
        });
}

}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::mobile::web::impl)
