#pragma once

#include "settings.h"
#include "mod_webpush/webpush_module.h"
#include <yxiva_mobile/ipusher.h>
#include <yxiva_mobile/error.h>
#include <yxiva_mobile/reports.h>
#include <ymod_webserver/expirable_stream.h>
#include <ymod_webserver/methods/default_answers.h>
#include <yplatform/find.h>
#include <memory>

namespace yxiva { namespace mobile { namespace web {

typedef settings::duration_type duration_type;

template <typename StreamPtr>
inline void send_intermediate_code(const StreamPtr& stream)
{
    stream->result(static_cast<ymod_webserver::codes::code>(245), "Meh", "");
}

template <>
inline void send_intermediate_code(const ymod_webserver::http::expirable_stream_ptr& stream)
{
    stream->store_intermediate_result(static_cast<ymod_webserver::codes::code>(245), "Meh");
}

inline std::pair<ymod_webserver::codes::code, string> get_response(
    const boost::system::error_code& ec)
{
    switch (to_error_condition(static_cast<error::code>(ec.value())))
    {
    case error::condition::success:
        return { ymod_webserver::codes::ok, "" };
    case error::condition::partial_success:
        return { static_cast<ymod_webserver::codes::code>(245), "" };
    case error::condition::bad_subscription:
        return { ymod_webserver::codes::reset_content, ec.message() };
    case error::condition::bad_request:
        return { ymod_webserver::codes::bad_request, ec.message() };
    case error::condition::internal_timeout:
        return { ymod_webserver::codes::gateway_timeout, ec.message() };
    case error::condition::cloud_error:
        return { ymod_webserver::codes::bad_gateway, ec.message() };
    case error::condition::cert_error:
        return { ymod_webserver::codes::forbidden, ec.message() };
    case error::condition::internal_error:
    default:
        return { ymod_webserver::codes::internal_server_error, ec.message() };
    }
}

template <typename StreamPtr>
inline void send_response(
    StreamPtr& stream,
    const boost::system::error_code& ec,
    const string& body = string(),
    const string& error_extra = string())
{
    if (ec || error_extra.size())
    {
        report_response_error(stream->ctx(), ec, error_extra);
    }
    if (ec.value() == error::intermediate_success)
    {
        send_intermediate_code(stream);
        return;
    }
    ymod_webserver::codes::code code;
    string default_body;
    std::tie(code, default_body) = get_response(ec);
    stream->result(code, code == 245 ? "Meh" : "", body.empty() ? default_body : body);
}

inline string pusher_module_name_to_platform_prefix(const char* pusher_module_name)
{
    string platform_prefix{ pusher_module_name };
    auto underscore_pos = platform_prefix.find_first_of('_');
    platform_prefix = platform_prefix.substr(0, underscore_pos);
    platform_prefix += "_";
    return platform_prefix;
}

struct webpush_proxy : public yplatform::log::contains_logger
{
    duration_type process_timeout;
    std::shared_ptr<webpush> mod_webpush;

    webpush_proxy(const duration_type& timeout) : process_timeout(timeout)
    {
        mod_webpush = yplatform::find<webpush, std::shared_ptr>("webpush");
    }

    void operator()(
        ymod_webserver::http::stream_ptr stream,
        const string& webpush_subscription,
        const string& payload,
        unsigned ttl,
        const string& service,
        const string& transit_id) const
    {
        report_new_request(stream->ctx(), service, transit_id);
        auto task = boost::make_shared<webpush_task_context>(stream->request());
        std::tie(task->subscription, task->payload, task->ttl, task->transit_id) =
            std::tie(webpush_subscription, payload, ttl, transit_id);

        call_pusher(
            ymod_webserver::http::make_expirable_stream(stream, process_timeout), task, "webpush");
    }

    void call_pusher(
        const ymod_webserver::http::expirable_stream_ptr& stream,
        webpush_task_context_ptr& task,
        const string& app_name) const
    {
        mod_webpush->push(
            task,
            [stream, task, app_name](const boost::system::error_code& ec, const string& extra) {
                send_response(stream, ec, {}, extra);
            });
    }
};

struct mobile_proxy : public yplatform::log::contains_logger
{
    duration_type process_timeout;
    unsigned long max_ttl;
    std::shared_ptr<ipusher> pusher;
    string platform_prefix;

    mobile_proxy(const char* pusher_module_name, const duration_type& timeout, unsigned long ttl)
        : process_timeout(timeout)
        , max_ttl(ttl)
        , platform_prefix(pusher_module_name_to_platform_prefix(pusher_module_name))
    {
        pusher = yplatform::find<ipusher, std::shared_ptr>(pusher_module_name);
    }

    void operator()(
        ymod_webserver::http::stream_ptr stream,
        const string& app_name,
        const string& token,
        const string& payload,
        unsigned ttl,
        const string& service,
        const string& transit_id) const
    {
        report_new_request_mobile(stream->ctx(), service, transit_id, platform_prefix + app_name);
        if (!pusher)
        {
            send_response(
                stream,
                make_error(error::module_not_configured),
                {},
                "pusher not configured: app=\"" + app_name + "\"");
            return;
        }

        auto task = boost::make_shared<mobile_task_context>(stream->request());
        std::tie(task->app_name, task->token, task->payload, task->ttl, task->transit_id) =
            std::tie(app_name, token, payload, ttl, transit_id);

        if (ttl > max_ttl)
        {
            stream->result(ymod_webserver::codes::bad_request, "invalid ttl value");
        }
        else
        {
            call_pusher(
                ymod_webserver::http::make_expirable_stream(stream, process_timeout),
                task,
                app_name);
        }
    }

    void operator()(
        ymod_webserver::http::stream_ptr stream,
        const string& app_name,
        const string& token,
        const string& service) const
    {
        report_new_request_mobile(stream->ctx(), service, "", app_name);
        if (!pusher)
        {
            send_response(
                stream,
                make_error(error::module_not_configured),
                {},
                "pusher not configured: app=\"" + app_name + "\"");
            return;
        }

        auto task = boost::make_shared<mobile_task_context>(stream->request());
        std::tie(task->app_name, task->token) = std::tie(app_name, token);

        call_check(
            ymod_webserver::http::make_expirable_stream(stream, process_timeout), task, app_name);
    }

    void call_pusher(
        const ymod_webserver::http::expirable_stream_ptr& stream,
        mobile_task_context_ptr& task,
        const string& app_name) const
    {
        pusher->push(
            task,
            [stream, task, app_name](const boost::system::error_code& ec, const string& body) {
                send_response(stream, ec, body, "app=" + app_name);
            });
    }

    void call_check(
        const ymod_webserver::http::expirable_stream_ptr& stream,
        mobile_task_context_ptr& task,
        const string& app_name) const
    {
        pusher->check(
            task,
            [stream, task, app_name](const boost::system::error_code& ec, const string& body) {
                send_response(stream, ec, body, "app=" + app_name);
            });
    }

    void update_application(const application_config& app) const
    {
        if (pusher)
        {
            pusher->update_application(app);
        }
    }
};

struct batch_proxy : public yplatform::log::contains_logger
{
    std::shared_ptr<batch_pusher> pusher;
    duration_type process_timeout;
    unsigned long max_ttl;

    batch_proxy(const char* pusher_module_name, const duration_type& timeout, unsigned long ttl)
        : process_timeout(timeout), max_ttl(ttl)
    {
        pusher = yplatform::find<batch_pusher, std::shared_ptr>(pusher_module_name);
    }

    void operator()(
        ymod_webserver::http::stream_ptr stream,
        const string& app_name,
        const string& payload,
        const string& tokens,
        unsigned ttl,
        const string& service,
        const string& transit_id) const
    {
        report_new_request(stream->ctx(), service, transit_id);
        auto task = boost::make_shared<batch_task_context>(stream->request());
        std::tie(task->app_name, task->payload, task->ttl, task->transit_id) =
            std::tie(app_name, payload, ttl, transit_id);

        if (auto error = task->tokens.parse(tokens))
        {
            stream->result(
                ymod_webserver::codes::bad_request, "invalid \"tokens\" field: " + *error);
            return;
        }
        if (!task->tokens.is_array())
        {
            return stream->result(
                ymod_webserver::codes::bad_request, "field \"tokens\" must be an array");
        }
        if (task->tokens.empty())
        {
            ymod_webserver::default_answers::send_no_argument_error(stream, "tokens");
            return;
        }
        for (auto&& token : task->tokens.array_items())
        {
            if (!token.is_string())
            {
                stream->result(
                    ymod_webserver::codes::bad_request,
                    "invalid \"tokens\" field: array of strings expected");
                return;
            }
        }

        if (ttl > this->max_ttl)
        {
            stream->result(ymod_webserver::codes::bad_request, "invalid ttl value");
        }
        else
        {
            call_pusher(
                ymod_webserver::http::make_expirable_stream(stream, process_timeout),
                task,
                app_name);
        }
    }

    void call_pusher(
        const ymod_webserver::http::expirable_stream_ptr& stream,
        batch_task_context_ptr& task,
        const string& app_name) const
    {
        pusher->push_batch(
            task,
            [stream, task, app_name](
                const boost::system::error_code& ec, json_value message, bool final) {
                if (final)
                {
                    if (ec.value() != error::success)
                    {
                        send_response(stream, ec, {}, "app=" + app_name);
                    }
                    else
                    {
                        stream->result(ymod_webserver::codes::ok, task->response.stringify());
                    }
                }
                else
                {
                    json_value val;
                    if (!message.is_null())
                    {
                        val["code"] = static_cast<int>(get_response(ec).first);
                        val["result"] = json_value(std::move(message));
                    }
                    else
                    {
                        // default response
                        auto [code, result] = get_response(ec);
                        val["code"] = static_cast<int>(code);
                        val["result"] = result;
                    }
                    task->response["results"].push_back(std::move(val));
                }
            });
    }
};

} // web
} // mobile
} // yxiva
