#pragma once

#include <yxiva_mobile/reports.h>
#include <yxiva/core/ec_crypto.h>
#include <yxiva/core/types.h>
#include <yplatform/encoding/base64.h>
#include <yplatform/time_traits.h>
#include <yplatform/log.h>
#include <boost/asio/io_service.hpp>
#include <vector>
#include <memory>
#include <mutex>
#include <stdexcept>

namespace yxiva { namespace mobile {

struct server_keys
{
    server_keys(
        std::vector<unsigned char>&& pub_key,
        string&& pub_key_b64,
        ec_crypto::evp_pkey_ptr keys,
        ec_crypto::ec_group_ptr eg)
        : public_key(std::move(pub_key))
        , public_key_base64(std::move(pub_key_b64))
        , keypair(std::move(keys))
        , ecgrp(std::move(eg))
    {
    }

    std::vector<unsigned char> public_key;
    // Used in request header, stored to avoid excessive encoding operations
    string public_key_base64;
    ec_crypto::evp_pkey_ptr keypair;
    ec_crypto::ec_group_ptr ecgrp;
};

struct encrypted_payload
{
    string data;
    string salt;
};

encrypted_payload encrypt_payload(
    const string& payload,
    const server_keys& keys,
    const string_view& client_auth_secret,
    const string_view& client_pub_key);

template <typename Sync>
class keys_store
    : public std::enable_shared_from_this<keys_store<Sync>>
    , public yplatform::log::contains_logger
{
public:
    // throws if generate failed
    keys_store(boost::asio::io_service& io, const yplatform::time_traits::duration& interval)
        : keys_(generate_keys()), timer_(io), interval_(interval)
    {
    }

    std::shared_ptr<server_keys> get()
    {
        std::lock_guard<Sync> lock(sync_);
        return keys_;
    }

    void run()
    {
        timer_.expires_from_now(interval_);
        auto self = this->shared_from_this();
        timer_.async_wait([self, this](const boost::system::error_code& ec) {
            if (!ec)
            {
                try
                {
                    auto new_keys = generate_keys();
                    {
                        std::lock_guard<Sync> lock(sync_);
                        std::swap(keys_, new_keys);
                    }
                }
                catch (const std::exception& ex)
                {
                    report_webpush_keys_generation_falied(logger(), ex.what());
                }
                report_webpush_keys_generated(logger());
                run();
            }
        });
    }

private:
    static std::shared_ptr<server_keys> generate_keys()
    {
        auto keypair = ec_crypto::generate_crypto_keys();
        ec_crypto::ec_key_ptr ec_key = ec_crypto::evp_to_ec(keypair);
        auto ecgrp = ec_crypto::make_ptr(EC_GROUP_dup(EC_KEY_get0_group(ec_key.get())));
        if (!ecgrp) throw std::runtime_error("error creating EC group");
        auto server_key = ec_crypto::get_public_key(ec_key);
        auto key_b64 = yplatform::base64_urlsafe_encode(server_key.begin(), server_key.end());
        return std::make_shared<server_keys>(
            std::move(server_key),
            string(key_b64.begin(), key_b64.end()),
            std::move(keypair),
            std::move(ecgrp));
    }

    Sync sync_;
    std::shared_ptr<server_keys> keys_;
    yplatform::time_traits::timer timer_;
    yplatform::time_traits::duration interval_;
};

}}
