#pragma once

#include <yxiva_mobile/reports.h>
#include <yxiva/core/ec_crypto.h>
#include <yxiva/core/types.h>
#include <yxiva/core/json.h>
#include <yplatform/encoding/base64.h>
#include <yplatform/time_traits.h>
#include <yplatform/range.h>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/asio/basic_deadline_timer.hpp>
#include <unordered_map>
#include <exception>
#include <chrono>
#include <utility>

namespace yxiva { namespace mobile {

struct vapid_record
{
    vapid_record(string&& auth_header, string key_header, time_t exp)
        : auth(std::move(auth_header)), key(std::move(key_header)), exp(exp)
    {
    }

    string auth;
    string key;
    time_t exp;
};

template <typename Sync>
class vapid_store
    : public std::enable_shared_from_this<vapid_store<Sync>>
    , public yplatform::log::contains_logger
{
    using duration = yplatform::time_traits::duration;
    using seconds = yplatform::time_traits::seconds;

    struct internal_keystore
    {
        internal_keystore(ec_crypto::evp_pkey_ptr&& k, string&& pk)
            : key(std::move(k)), pub_key(std::move(pk))
        {
        }

        ec_crypto::evp_pkey_ptr key;
        string pub_key;
    };

public:
    vapid_store(const duration& valid_for, const duration& margin)
        : valid_for_(yplatform::time_traits::duration_cast<seconds>(valid_for).count())
        , valid_for_margin_(yplatform::time_traits::duration_cast<seconds>(margin).count())
    {
    }

    std::shared_ptr<vapid_record> get(const string& origin)
    {
        std::lock_guard<Sync> lock(sync_);
        auto vapid = vapids_.find(origin);
        if (vapid != vapids_.end() && valid_with_margin(*vapid->second)) return vapid->second;
        auto new_vapid = gen_vapid(origin);
        vapids_.insert(vapid, std::make_pair(origin, new_vapid));
        return new_vapid;
    }

    bool reset(const string& public_key_file, const string& contact)
    {
        try
        {
            auto key_ptr = ec_crypto::read_pem(public_key_file);
            auto ec_key = ec_crypto::evp_to_ec(key_ptr);
            auto public_key = ec_crypto::get_public_key(ec_key);
            if (public_key.empty())
            {
                throw std::runtime_error("failed to extract public key - invalid vapid key");
            }
            auto b64_sz = (public_key.size() * 4) / 3 + (public_key.size() % 3) ? 1 : 0;
            string public_b64;
            public_b64.reserve(b64_sz);
            public_b64 += yplatform::base64_urlsafe_encode(public_key.begin(), public_key.end());
            auto new_keys =
                std::make_shared<internal_keystore>(std::move(key_ptr), std::move(public_b64));
            {
                std::lock_guard<Sync> lock(sync_);
                vapid_sub_ = contact;
                keys_ = new_keys;
                vapids_.clear();
            }
            report_vapid_keys_ready(logger(), public_b64);
            return true;
        }
        catch (const std::exception& e)
        {
            report_vapid_keys_load_falied(logger(), e.what());
        }
        return false;
    }

private:
    std::shared_ptr<vapid_record> gen_vapid(const string& origin)
    {
        // FCM is said to be excessively sensitive to the vapid header.
        // It's also always the same. String below is b64 for {"typ":"JWT","alg":"ES256"}
        static const string header_b64 = "eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzI1NiJ9.";
        try
        {
            if (!keys_) throw std::runtime_error("vapid key not available");

            auto start_point = yplatform::time_traits::clock::now();
            uint64_t expiration_ts = std::time(nullptr) + valid_for_;

            json_value vapid_claim_json;
            vapid_claim_json["aud"] = origin;
            vapid_claim_json["sub"] = vapid_sub_;
            vapid_claim_json["exp"] = expiration_ts;

            auto claim = vapid_claim_json.stringify();

            string vapid;
            vapid.reserve(header_b64.size() + (claim.size() * 4) / 3 + 88);
            vapid += header_b64;
            vapid += yplatform::base64_urlsafe_encode(claim.begin(), claim.end());

            auto sign = ec_crypto::sign(vapid.data(), vapid.size(), keys_->key);
            vapid += ".";
            vapid += yplatform::base64_urlsafe_encode(sign.begin(), sign.end());

            auto time_run = yplatform::time_traits::clock::now() - start_point;

            report_vapid_generated(logger(), origin, vapid, time_run);
            return std::make_shared<vapid_record>(std::move(vapid), keys_->pub_key, expiration_ts);
        }
        catch (const std::exception& ex)
        {
            report_vapid_generation_failed(logger(), origin, ex.what());
            throw;
        }
    }

    bool valid_with_margin(const vapid_record& vapid) const
    {
        return std::time(nullptr) + valid_for_margin_ < vapid.exp;
    }

    Sync sync_;
    std::unordered_map<string, std::shared_ptr<vapid_record>> vapids_;
    string vapid_sub_;
    // For how long vapid is valid since generation in seconds.
    unsigned int valid_for_;
    // Time before expiration when we should generate new vapid in seconds.
    unsigned int valid_for_margin_;
    std::shared_ptr<internal_keystore> keys_;
};

}}
