#pragma once

#include "notification_types.h"
#include <yxiva_mobile/error.h>
#include <yxiva/core/types.h>
#include <map>
#include <memory>
#include <vector>

namespace yxiva { namespace mobile {

struct chunk
{
    string payload;
    wns_notification_type::type type;

    bool operator==(const chunk& other) const
    {
        return payload == other.payload && type == other.type;
    }
};

template <typename DataT, typename Url, typename ConvertFunc>
inline std::shared_ptr<DataT> make_chunked_data(
    const Url& request_url,
    const string& raw_payload,
    ConvertFunc converter)
{
    static const size_t MAX_RAW_LENGTH = 5000; // cloud limit
    static const string EMPTY;
    static const string XNAMES[wns_notification_type::COUNT] = {
        "x-toast", EMPTY, "x-tile", "x-badge"
    };

    auto chunk_data = std::make_shared<DataT>();

    for (int type = 0; type < wns_notification_type::COUNT; ++type)
    {
        const auto& value = (type == wns_notification_type::raw) ?
            raw_payload :
            request_url.param_value(XNAMES[type], EMPTY);

        if (value.empty()) continue;

        if (type == wns_notification_type::raw && value.size() > MAX_RAW_LENGTH)
        {
            chunk_data->chunk_results.push_back(error::code::invalid_payload_length);
            continue;
        }
        auto enum_type = static_cast<wns_notification_type::type>(type);
        string notification;
        if (auto converted = converter(value, enum_type, notification))
        {
            chunk_data->chunks.push_back(chunk{ std::move(notification), enum_type });
        }
        else
        {
            // TODO log convert error
            chunk_data->chunk_results.push_back(error::code::invalid_payload);
        }
    }

    if (chunk_data->chunks.empty() && chunk_data->chunk_results.empty())
    {
        chunk_data->chunk_results.push_back(error::code::invalid_payload);
    }

    return chunk_data;
}

}}
