#include "convert.h"
#include <yxiva_mobile/reports.h>
#include <yxiva/core/types.h>
#include <yxiva/core/json.h>
#include <yxiva/core/quote_xml.h>
#include <yplatform/util/sstream.h>
#include <exception>
#include <unordered_map>

namespace yxiva::mobile {
namespace {

using formatters::quoteXML;
using ct = std::char_traits<char>;
using sstream = yplatform::sstream;
using handler_type = void (*)(sstream&, unsigned, const string&);

inline json_value json_parse(const string& source)
{
    json_value ret;
    ret.parse(source);
    return ret;
}

const string HANDLER_IMAGE = "special_image";
const string HANDLER_TEXT = "special_text";
const json_value XML_STRUCTURE = json_parse(
    R"({"visual":{"binding":{"text":")" + HANDLER_TEXT + R"(","image": ")" + HANDLER_IMAGE +
    R"("}},"audio":{}})");
const char* XML_TAG_BADGE = "badge";
const char* WNS_XML_TAG_TILE = "tile";
const char* WNS_XML_TAG_TOAST = "toast";

// Handlers for text and image tags
void write_text(sstream& xml, unsigned id, const string& text)
{
    xml << "<text id=\"" << id << "\">" << quoteXML(text) << "</text>";
}

void write_image(sstream& xml, unsigned id, const string& src)
{
    xml << "<image id=\"" << id << "\" src=\"" << quoteXML(src) << "\"/>";
}

const std::unordered_map<string, handler_type> HANDLERS = {
    { HANDLER_TEXT, &write_text },
    { HANDLER_IMAGE, &write_image },
    { "", [](sstream&, unsigned, const string&) {} }
};

// counts "special fields" and writes them with specific handler
void handle_special_object(sstream& xml, const json_value& object, handler_type handler)
{
    unsigned counter = 1;
    if (object.is_array())
    {
        for (auto&& val : object.array_items())
        {
            auto text_val = val.to_string("");
            handler(xml, counter++, text_val);
        }
    }
    else
    {
        auto text_val = object.to_string("");
        handler(xml, counter, text_val);
    }
}

void write_attrs(sstream& xml, const json_value& root, const json_value& structure)
{
    if (!root.is_object()) return;
    for (auto field = root.members_begin(); field != root.members_end(); ++field)
    {
        if (structure.has_member(field.key())) continue;
        // if json field is not xml child node - write it as an attribute
        xml << " " << quoteXML(field.key()) << "=\"" << quoteXML((*field).to_string("")) << "\"";
    }
}

void convert_object(
    sstream& xml,
    const json_value& root,
    const string& tag,
    const json_value& structure)
{
    // can't handle non-object in this function
    if (!root.is_object())
    {
        report_wns_invalid_payload(task_context::fake(), "wrong element type " + root.stringify());
        return;
    }
    // Open tag and write attributes
    xml << "<" << tag;
    write_attrs(xml, root, structure);
    bool hasChild = false;

    // check structure to see if there are any child nodes
    for (auto field = structure.members_begin(); field != structure.members_end(); ++field)
    {
        auto name = field.key();
        auto child = root.cref[name];
        if (!child) continue;
        // terminate open tag if the node has children
        if (!hasChild)
        {
            hasChild = true;
            xml << ">";
        }
        // hack - string children mark special handlers to write "counted" fields
        if ((*field).is_string())
        {
            auto special_handler = (*field).to_string("");
            handle_special_object(xml, child, HANDLERS.at(special_handler));
            // if it's not special child - write one or collection of the same tags
        }
        else if (child.is_object())
        {
            convert_object(xml, child, string(name), *field);
        }
        else if (child.is_array())
            for (auto&& val : child.array_items())
                convert_object(xml, val, string(name), *field);
    }

    // write close tag or terminate open tag if there are no children
    if (hasChild)
    {
        xml << "</" << quoteXML(tag) << ">";
    }
    else
    {
        xml << "/>";
    }
}

operation::result convert_default(
    string& xml_string,
    const string& data,
    wns_notification_type::type type)
{
    static const string xml_attr_template = " template=\"ToastText01\"";
    static const string xml_attr_value = " value=\"";
    static const string xml_tag_binding = "binding";
    static const string xml_tag_text = "text";
    static const string xml_tag_visual = "visual";
    static const string::size_type badge_base_len =
        4 + ct::length(XML_TAG_BADGE) + xml_attr_value.size();
    static const string::size_type toast_base_len = 27 +
        2 *
            (ct::length(WNS_XML_TAG_TOAST) + xml_tag_visual.size() + xml_tag_binding.size() +
             xml_tag_text.size()) +
        xml_attr_template.size();

    switch (type)
    {
    case wns_notification_type::toast:
    {
        sstream(xml_string, toast_base_len + data.size())
            << "<" << WNS_XML_TAG_TOAST << "><" << xml_tag_visual << "><" << xml_tag_binding
            << xml_attr_template << "><" << xml_tag_text << " id=\"1\""
            << ">" << quoteXML(data) << "</" << xml_tag_text << "></" << xml_tag_binding << "></"
            << xml_tag_visual << "></" << WNS_XML_TAG_TOAST << ">";
        break;
    }
    case wns_notification_type::badge:
    {
        sstream(xml_string, badge_base_len + data.size())
            << "<" << XML_TAG_BADGE << xml_attr_value << quoteXML(data) << "\"/>";
        break;
    }
    default:
        return "default conversion not supported for this notification type";
    }
    return operation::success;
}

operation::result tag_name_from_type(wns_notification_type::type type, string& tag)
{
    switch (type)
    {
    case wns_notification_type::badge:
        tag = XML_TAG_BADGE;
        break;
    case wns_notification_type::tile:
        tag = WNS_XML_TAG_TILE;
        break;
    case wns_notification_type::toast:
        tag = WNS_XML_TAG_TOAST;
        break;
    default:
        return "notification type not supported";
    }
    return operation::success;
}
}

operation::result convert_wns(
    const string& input,
    wns_notification_type::type notification_type,
    string& output)
{
    if (notification_type == wns_notification_type::raw)
    {
        output = input;
        return operation::success;
    }
    json_value root;
    auto error = root.parse(input);
    // if can't parse json, construct default xml for given
    // push type and input string
    if (error || root.is_string() || root.is_number())
    {
        return convert_default(output, input, notification_type);
    }
    if (!root.is_object())
    {
        return "invalid json: need object to convert to xml";
    }
    try
    {
        yplatform::sstream xml(output, 2 * input.size());
        // convert starting with "base" node
        string tag_name;
        auto tag_res = tag_name_from_type(notification_type, tag_name);
        if (!tag_res) return tag_res;
        convert_object(xml, root, tag_name, XML_STRUCTURE);
        return operation::success;
    }
    catch (const std::exception& e)
    {
        return e.what();
    }
}

}
