#pragma once

#include "access_tokens.h"
#include "chunks.h"
#include "convert.h"
#include <yxiva_mobile/error.h>
#include <yxiva_mobile/ipusher.h>
#include <yxiva_mobile/reports.h>
#include <yxiva/core/platforms.h>
#include <ymod_httpclient/call.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/spinlock.h>
#include <yplatform/util/sstream.h>
#include <vector>

namespace yxiva { namespace mobile {

using yplatform::sstream;
namespace p = std::placeholders;

struct wns_settings
{
    struct app_data
    {
        string name;
        string sid;
        string key;
    };

    string wns_url;
    yplatform::time_traits::duration auth_timeout;
    yplatform::time_traits::duration send_timeout;
    std::vector<app_data> test_apps;

    void load(const yplatform::ptree& conf)
    {
        wns_url = conf.get<string>("wns_url");
        auth_timeout = conf.get<yplatform::time_traits::duration>("auth_timeout");
        send_timeout = conf.get<yplatform::time_traits::duration>("send_timeout");
        yplatform::ptree empty;
        for (auto& app : conf.get_child("test_apps", empty))
        {
            if (app.first == "app")
                test_apps.push_back(app_data{ app.second.get<string>("name"),
                                              app.second.get<string>("sid"),
                                              app.second.get<string>("key") });
        }
    }
};

class mod_wns
    : public ipusher
    , public yplatform::module
{
    using spinlock = yplatform::spinlock;
    using scoped_lock = std::unique_lock<spinlock>;
    using access_token_storage_t = access_token_storage<async_get_access_token_op>;

    struct request_data
    {
        mobile_task_context_ptr ctx;
        size_t position = 0;
        std::vector<chunk> chunks;
        std::vector<error::code> chunk_results;
        string access_token;
        callback_t callback;
    };

public:
    void init(const yplatform::ptree& conf)
    {
        settings_.load(conf);
        httpclient_ = yplatform::find<yhttp::call, std::shared_ptr>("http_client");
        for (auto& app : settings_.test_apps)
        {
            add_app_token(app.name, app.sid, app.key);
        }
    }

    void push(mobile_task_context_ptr ctx, callback_t&& cb) override
    {
        scoped_lock lock(tokens_guard_);
        auto it = app_tokens_.find(ctx->app_name);
        if (it == app_tokens_.end())
        {
            lock.unlock();
            cb(make_error(error::no_cert));
            return;
        }
        auto token_storage = it->second;
        lock.unlock();

        token_storage->async_get(std::bind(
            &mod_wns::handle_get_access_token,
            shared_from(this),
            p::_1,
            p::_2,
            ctx,
            std::move(cb)));
    }

    void check(mobile_task_context_ptr /*ctx*/, callback_t&& cb) override
    {
        cb(make_error(error::not_implemented));
    }

    void update_application(const application_config& conf) override
    {
        auto credential = yxiva::wns::parse_secret_key(conf.secret_key);

        scoped_lock lock(tokens_guard_);
        // erase app only if both sid and secret_key are empty
        if (conf.secret_key.empty())
        {
            app_tokens_.erase(conf.app_name);
        }
        else
        {
            add_app_token(conf.app_name, credential.sid, credential.secret);
        }

        lock.unlock();
        report_wns_credentials_found(logger(), "update", conf.app_name);
    }

    // get_stats?
    // it's a bad idea to get stats for all apps
    // if there are too many apps in the module

private:
    void handle_get_access_token(
        error::code ec,
        const string& access_token,
        mobile_task_context_ptr& ctx,
        callback_t& cb)
    {
        if (ctx->is_cancelled()) return;

        if (ec)
        {
            cb(make_error(ec));
            return;
        }

        auto chunked_data =
            make_chunked_data<request_data>(ctx->request->url, ctx->payload, convert_wns);

        chunked_data->ctx = ctx;
        chunked_data->access_token = access_token;
        chunked_data->callback = std::move(cb);

        process_chunks(chunked_data);
    }

    void process_chunks(std::shared_ptr<request_data>& chunked_data)
    {
        if (chunked_data->position < chunked_data->chunks.size())
        {
            if (is_partial_success(chunked_data))
            {
                chunked_data->callback(make_error(error::intermediate_success));
            }
            send(chunked_data);
        }
        else
        {
            chunked_data->callback(make_error(calc_mixed_result(chunked_data->chunk_results)));
        }
    }

    bool is_partial_success(std::shared_ptr<request_data>& chunked_data)
    {
        bool result = false;
        for (auto& ec : chunked_data->chunk_results)
        {
            if (ec == error::success)
            {
                result = true;
                break;
            }
        }
        return result;
    }

    void send(std::shared_ptr<request_data>& chunked_data)
    {
        assert(chunked_data->position < chunked_data->chunks.size());

        static const string XHEADERS[wns_notification_type::COUNT] = {
            "X-WNS-Type: wns/toast\r\n",
            "X-WNS-Type: wns/raw\r\nX-WNS-Cache-Policy: cache\r\n",
            "X-WNS-Type: wns/tile\r\n",
            "X-WNS-Type: wns/badge\r\n"
        };
        static const string COMMON_HEADERS = "X-WNS-RequestForStatus: true\r\n"
                                             "Authorization: Bearer ";

        // TODO additional headers for future support:
        // X-WNS-SuppressPopup - bright
        // X-WNS-Group - toast group

        auto& chunk = chunked_data->chunks[chunked_data->position];
        string headers;
        sstream(headers, 800) << XHEADERS[chunk.type] << COMMON_HEADERS
                              << chunked_data->access_token << "\r\n"
                              << (chunk.type == wns_notification_type::raw ?
                                      "Content-Type: application/octet-stream\r\n" :
                                      "Content-Type: text/xml\r\n")
                              << "X-WNS-TTL: " << chunked_data->ctx->ttl << "\r\n";

        yhttp::options opts;
        opts.timeouts.total = settings_.send_timeout;
        httpclient_->async_run(
            chunked_data->ctx,
            yhttp::request::POST(chunked_data->ctx->token, headers, string(chunk.payload)),
            opts,
            std::bind(&mod_wns::handle_sent, shared_from(this), p::_1, p::_2, chunked_data));
    }

    void handle_sent(
        const boost::system::error_code& ec,
        yhttp::response response,
        std::shared_ptr<request_data>& chunked_data)
    {
        auto& ctx = chunked_data->ctx;

        if (ctx->is_cancelled()) return;

        auto result_code = ec ? error::network_error : http_status_to_error_code(response.status);

        if (result_code)
        {
            report_wns_request_error(
                ctx, ec, std::to_string(response.status) + " " + response.body);
            chunked_data->callback(make_error(result_code));
            return;
        }

        // TODO log?
        /*
          X-DeviceConnectionStatus: disconnected
          X-NotificationStatus: Received
          X-SubscriptionStatus: Active
          X-WnsMessageID|X-WNS-MSG-ID
        */

        chunked_data->chunk_results.push_back(result_code);

        chunked_data->position++;
        process_chunks(chunked_data);
    }

    error::code http_status_to_error_code(int status)
    {
        error::code result_code;

        if (status / 100 == 2)
        {
            result_code = error::success;
        }
        else if (status == 401 || status == 403)
        {
            result_code = error::invalid_cert;

            // not sure about 400 - no way to detect it's a bad URI or bad request
        }
        else if (status == 400 || status == 404 || status == 410)
        {
            result_code = error::invalid_subscription;
        }
        else if (status == 413)
        {
            result_code = error::invalid_payload_length;
        }
        else
        {
            result_code = error::cloud_error;
        }
        return result_code;
    }

    void add_app_token(const string& name, const string& sid, const string& key)
    {
        auto& token_storage = app_tokens_[name];
        if (!token_storage)
        {
            async_get_access_token_op get_token_op;
            get_token_op.auth_url = settings_.wns_url;
            get_token_op.timeout = settings_.auth_timeout;
            get_token_op.httpclient = httpclient_;
            get_token_op.name = name;
            token_storage = std::make_shared<access_token_storage_t>(get_token_op, sid, key);
        }
        else
        {
            token_storage->reset(sid, key);
        }
    }

    wns_settings settings_;
    spinlock tokens_guard_;
    std::shared_ptr<yhttp::call> httpclient_;
    std::unordered_map<string, std::shared_ptr<access_token_storage_t>> app_tokens_;
};

}}
