#include <ymod_httpclient/call.h>
#include <ymod_webserver/server.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/loader.h>
#include <yplatform/encoding/url_encode.h>
#include <yplatform/util/sstream.h>
#include <string>
#include <memory>

namespace ymailtest {

class module: public yplatform::module
{
public:
    void init (const yplatform::ptree& config);
private:
    void process_request(ymod_webserver::http::stream_ptr stream, std::string token);
    void process_response(const boost::system::error_code& err, yhttp::response response, ymod_webserver::http::stream_ptr stream);

    std::string server;
    std::string subscribe_token;
    std::string send_token;
    std::shared_ptr<yhttp::call> http_call;
};

void module::init (const yplatform::ptree& config)
{
    http_call = yplatform::find<yhttp::call, std::shared_ptr>("http_client");
    auto webserver = yplatform::find<ymod_webserver::server, std::shared_ptr>("web_server");
    server = config.get<std::string>("server");
    subscribe_token = config.get<std::string>("subscribe_token");
    send_token = config.get<std::string>("send_token");

    webserver->bind("", {"/v2/send"},
      std::bind(&module::process_request, this, std::placeholders::_1, send_token));
    webserver->bind("", {"/v2/subscribe/app"},
      std::bind(&module::process_request, this, std::placeholders::_1, subscribe_token));
    webserver->bind("", {"/v2/unsubscribe/app"},
      std::bind(&module::process_request, this, std::placeholders::_1, subscribe_token));
    webserver->bind("", {"/v2/subscriptions/user"},
      std::bind(&module::process_request, this, std::placeholders::_1, subscribe_token));
    webserver->bind("", {"/v2/apns_queue_repeat"},
      std::bind(&module::process_request, this, std::placeholders::_1, subscribe_token));
}

void module::process_request(ymod_webserver::http::stream_ptr stream, std::string token)
{
    auto method = stream->request()->method;
    auto url = server + stream->request()->url.make_full_path();
    yplatform::sstream ss(url);
    ss << "?token=" << token;
    for (auto&& param: stream->request()->url.params) {
      if (param.second.size())
        ss << "&" << param.first << "=" << yplatform::url_encode(param.second);
    }
    L_(info) << url;

    if (method == ymod_webserver::methods::mth_post) {
        auto& body = stream->request()->raw_body;
        http_call->async_run(stream->ctx(),
        yhttp::request::POST(url, std::string(body.begin(), body.end())),
            [this, stream] (const boost::system::error_code& err, yhttp::response response) {
                process_response(err, response, stream);
            }
        );
    } else if (method == ymod_webserver::methods::mth_get) {
        http_call->async_run(stream->ctx(),
        yhttp::request::GET(url),
            [this, stream] (const boost::system::error_code& err, yhttp::response response) {
                  process_response(err, response, stream);
            });
    } else
        stream->set_code(ymod_webserver::codes::method_not_allowed);
}

void module::process_response(const boost::system::error_code& err, yhttp::response response, ymod_webserver::http::stream_ptr stream)
{
    if (err)
        stream->result(ymod_webserver::codes::bad_gateway, err.message());
    else
        stream->result(static_cast<ymod_webserver::codes::code>(response.status), response.body);
}

}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(ymailtest::module)

int main(int argc, char* argv[])
{
    if (argc != 2) {
        std::cout << "usage " << argv[0] << " <config>\n";
        return 1;
    }

    return yplatform_start(argv[1]);
}