#pragma once

#include <vector>
#include <deque>

#include <boost/function.hpp>

#include "directed_buffer.h"
#include "window.h"
#include "stream_settings.h"

namespace pipeline {

/**
 * Stream is supposed to be piped with other streams, so BaseStream provides several types of operations:
 * 1) buffer management and access to stored data (put/commit/consume/at)
 * 2) observation of size of data that can be consumed (on_free_space_for_consume)
 */
template<typename Data>
class BaseStream
{
    typedef BaseStream<Data> this_t;
    typedef DirectedBuffer<Data> buffer_t;

public:
    typedef std::vector<Data> temp_collection_t;
    typedef std::shared_ptr<temp_collection_t> temp_collection_ptr;

    enum class Event { None, ReadyForConsume, NewDataReceived };

    struct stats
    {
        std::size_t buffer_size = 0U;
        std::size_t total_commited_size = 0U;
        std::size_t begin_id = 0U;
    };

    BaseStream(const StreamSettings& settings)
    : settings_(settings),
      buffer_(settings_.capacity),
      window_(0, settings_.window),
      current_allowed_consume_size_(0),
      consume_any_size_(true)
    {}

    template <template <typename...> class Collection>
    void put_range(std::shared_ptr<Collection<Data>> data)
    {
        if (data->empty()) return;

        buffer_.put_range(*data);
        stats_.buffer_size = buffer_.size();
    }

    void commit(std::size_t id)
    {
        buffer_.commit(id);
        stats_.total_commited_size++;
    }

    template <template <typename...> class Collection>
    void commit_all(const Collection<std::size_t>& ids)
    {
        buffer_.commit_all(ids);
        stats_.total_commited_size += ids.size();
    }

    /**
     * commit_until and commit shouldn't be used simultaneously.
     * @todo: fix this or add restrictions to code
     */
    void commit_until(std::size_t id)
    {
        buffer_.commit_until(id);
        stats_.total_commited_size = buffer_.committed_size();
    }

    void on_free_space_for_consume(std::size_t free_space) // free_space is allowed to be 0
    {
        current_allowed_consume_size_ += free_space;
    }

    Event next_event()
    {
        window_.end_id(buffer_.end_id(), stats_.total_commited_size);
        if (is_consume_available()) return Event::ReadyForConsume;
        if (window_.next_interval_available()) return Event::NewDataReceived;
        return Event::None;
    }

    bool is_consume_available() const
    {
        std::size_t commited_size = buffer_.committed_size();
        if (!commited_size) return false;

        std::size_t not_commited_size = buffer_.size() - commited_size;
        return (commited_size >= settings_.grouped_flush_size || not_commited_size == 0U)
            && (consume_any_size_ || current_allowed_consume_size_ > 0);
    }

    temp_collection_ptr consume()
    {
        if (!is_consume_available()) {
            #ifdef PIPELINE_DEBUG
                YLOG_G(debug) << "consume not available";
            #endif
            return nullptr;
        }

        temp_collection_ptr consumed_data = nullptr;
        consumed_data = on_consumed( consume_any_size_ ? buffer_.consume()
                                                       : buffer_.consume(current_allowed_consume_size_));
        window_.begin_id(buffer_.begin_id());
        return consumed_data;
    }

    /**
     * Getters
     */
    const Data& operator[](std::size_t id) const
    {
        return buffer_.operator[](id);
    }

    std::size_t free_space() const { return buffer_.free_space(); }

    /**
     * Returns next interval of new data.
     * Can be used while processing NewDataReceived for informing only about new ids in window.
     */
    interval_t next_interval()
    {
        auto interval = window_.next_interval();
        window_.end_id(buffer_.end_id(), stats_.total_commited_size);
        return interval;
    }

    void reset_consume_size_observer(bool is_fixed_size)
    {
        consume_any_size_ = !is_fixed_size;
        current_allowed_consume_size_ = 0;
    }

    /**
     * Stats getters
     */
    std::size_t buffer_size() const { return stats_.buffer_size; }
    std::size_t total_commited_size() const { return stats_.total_commited_size; }
    std::size_t begin_id() const { return buffer_.begin_id(); }
    std::size_t end_id() const { return buffer_.end_id(); }

private:
    temp_collection_ptr on_consumed(temp_collection_ptr splice)
    {
        if (splice && not splice->empty()) {
            if (!consume_any_size_) current_allowed_consume_size_ -= splice->size();

            stats_.buffer_size = buffer_.size();
            stats_.total_commited_size -= splice->size();
            stats_.begin_id = buffer_.begin_id();
            return splice;
        }
        return nullptr;
    }

private:
    StreamSettings settings_;
    buffer_t buffer_;
    Window window_;

    std::size_t current_allowed_consume_size_;
    bool consume_any_size_;

    stats stats_;
};

}
