#pragma once

#include <string>
#include <deque>
#include <vector>
#include <cassert>
#include <numeric>

using std::string;

namespace pipeline {

template<typename Data>
class DirectedBuffer
{
public:
    typedef std::deque<Data> buffer_t;
    typedef typename buffer_t::iterator iterator_t;
    typedef typename buffer_t::const_iterator const_iterator_t;

    DirectedBuffer(std::size_t capacity)
    : capacity_(capacity),
      begin_id_(0),
      id_committed_end_(0)
    {
    }

    template <template <typename...> class Collection>
    void put_range(const Collection<Data>& data)
    {
        size_t insert_size = data.size();
        if (insert_size == 0) return;

        size_t current_size = size();
        size_t new_size = current_size + insert_size;
        assert(new_size <= capacity_);

        buffer_.insert(buffer_.end(), data.begin(), data.end());
        committed_state_.resize(new_size, false);
    }

    template <typename InputIter>
    InputIter put_range(const InputIter& data_begin, const InputIter& data_end)
    {
        for (auto it = data_begin; it != data_end; it++) {
            if (not put(*it)) return it;
        }
        return data_end;
    }

    void commit(std::size_t id)
    {
        if (id < id_committed_end_) return;

        std::size_t pos = id2position(id);
        committed_state_[pos] = true;

        move_forward_committed_end(id);
    }

    template <template <typename...> class Collection>
    void commit_all(const Collection<std::size_t>& ids)
    {
        std::size_t min_commited_id = ids.front();
        for (std::size_t id : ids) {
            if (min_commited_id > id) min_commited_id = id;

            if (id < id_committed_end_) continue;

            std::size_t pos = id2position(id);
            committed_state_[pos] = true;
        }
        move_forward_committed_end(min_commited_id);
    }

    void commit_until(std::size_t last_id)
    {
        if (id_committed_end_ > last_id) return;

        std::size_t begin_pos = id2position(id_committed_end_);
        std::size_t end_pos = id2position(last_id) + 1;
        std::fill(committed_state_.begin() + begin_pos, committed_state_.begin() + end_pos, true);

        id_committed_end_ = last_id + 1;
    }

    void clear()
    {
        begin_id_ = 0;
        id_committed_end_ = 0;
        buffer_.clear();
        committed_state_.clear();
    }

    template <template <typename...> class Collection = std::vector>
    std::shared_ptr<Collection<Data>> consume()
    {
        return consume<Collection>(committed_size());
    }

    template <template <typename...> class Collection = std::vector>
    std::shared_ptr<Collection<Data>> consume(std::size_t count)
    {
        std::size_t splice_count = std::min(committed_size(), count);
        if (splice_count) {
            auto end = begin() + splice_count;
            std::shared_ptr<Collection<Data>> splice = std::make_shared<Collection<Data>>(begin(), end);
            buffer_.erase(begin(), end);
            committed_state_.erase(committed_state_.begin(), committed_state_.begin() + splice_count);
            begin_id_ += splice_count;
            return splice;
        }
        return nullptr;
    }

    const Data& operator[](std::size_t id) const
    {
        std::size_t position = id2position(id);
        return buffer_[position];
    }

    const Data& at(std::size_t id) const
    {
        return operator[](id);
    }

    std::size_t committed_end() const { return id_committed_end_; }
    std::size_t free_space() const { return capacity_ - buffer_.size(); }

    std::size_t capacity() const { return capacity_; }
    std::size_t size() const { return buffer_.size(); }
    bool empty() const { return buffer_.empty(); }

    iterator_t begin() { return buffer_.begin(); }
    iterator_t end() { return buffer_.end(); }
    const_iterator_t begin() const { return buffer_.begin(); }
    const_iterator_t end() const { return buffer_.end(); }

    std::size_t begin_id() const { return begin_id_; }
    std::size_t end_id() const { return begin_id_ + size(); }

    std::size_t committed_size() const { return id_committed_end_ - begin_id_; }
    iterator_t committed_end_it() { return begin() + committed_size(); }
    const_iterator_t committed_end_it() const { return begin() + committed_size(); }
    std::size_t full_commited_size() const { return std::accumulate(committed_state_.begin(), committed_state_.end(), 0); }

protected:
    std::size_t id2position(std::size_t id) const
    {
        if (id < begin_id_)
            throw std::out_of_range("id < committed offset");

        std::size_t pos = id - begin_id_;
        if (pos >= buffer_.size())
            throw std::out_of_range("id > max stored id");

        return pos;
    }

    bool put(const Data& data)
    {
        if (buffer_.size() >= capacity_) {
            return false;
        }

        buffer_.push_back(data);
        committed_state_.push_back(false);
        return true;
    }

    void move_forward_committed_end(std::size_t begin_id)
    {
        if (begin_id == id_committed_end_) {
            id_committed_end_++;
            for (std::size_t i = id_committed_end_ - begin_id_; i < size() && committed_state_[i]; i++) {
                id_committed_end_++;
            }
        }
    }

private:
    std::size_t capacity_;
    buffer_t buffer_;
    std::deque<bool> committed_state_;
    std::size_t begin_id_;
    std::size_t id_committed_end_;
};

}
