#pragma once

#include <string>
#include <boost/bind.hpp>
#include <boost/asio.hpp>

#include "progressive_retrier.h"

namespace pipeline {

/**
 * Can be used to generate stream's data.
 * Generator is supposed to be connected to the first stream in pipeline.
 */
template <typename StreamT>
class Generator: public std::enable_shared_from_this<Generator<StreamT>>
{
    typedef Generator<StreamT> this_t;
    typedef typename StreamT::this_ptr stream_ptr;

    typedef std::shared_ptr<ProgressiveRetrier> retrier_ptr;

public:
    Generator(boost::asio::io_service& io, std::size_t min_free_space,
              const ProgressiveRetrier::interval_t& interval)
    : Generator(io, min_free_space, ProgressiveRetrier::intervals_t{ interval })
    {}

    Generator(boost::asio::io_service& io, std::size_t min_free_space = 1U,
              const ProgressiveRetrier::intervals_t& intervals = { time_traits::seconds(2) })
    : min_free_space_(min_free_space),
      stopped_(true),
      generate_active_(false),
      retrier_(std::make_shared<ProgressiveRetrier>(io, intervals)),
      available_free_space_(0)
    {}

    virtual ~Generator()
    {}

    stream_ptr output() { return output_; }

    void output(stream_ptr out)
    {
        assert(not output_);
        output_ = out;
    }

    virtual void pause()
    {
        scoped_lock guard(mutex_);
        stopped_ = true;
        retrier_->stop();
    }

    virtual void resume()
    {
        scoped_lock guard(mutex_);
        stopped_ = false;
        on_free_space_impl(guard);
    }

    virtual void start()
    {
        scoped_lock guard(mutex_);
        stopped_ = false;
        available_free_space_ = 0;
        auto stream = output_;
        guard.unlock();
        retrier_->start(boost::bind(&this_t::on_free_space, this->shared_from_this(), 0));

        if (stream) {
            stream->reset_free_space_handler(
                boost::bind(&this_t::on_free_space, this->shared_from_this(), _1)
            );
        }
    }

    virtual void stop()
    {
        scoped_lock guard(mutex_);
        auto stream = output_;
        stopped_ = true;
        output_.reset();
        guard.unlock();
        retrier_->stop();

        if (stream) stream->reset_free_space_handler();
    }

    bool is_stopped() {
        scoped_lock guard(mutex_);
        return stopped_;
    }

protected:
    void on_free_space(std::size_t free_space)
    {
        scoped_lock guard(mutex_);
        on_free_space_impl(guard, free_space);
    }

    /**
    * If free_space == 0 it is supposed to be retry.
    */
    void on_free_space_impl(scoped_lock& guard, std::size_t free_space = 0)
    {
        available_free_space_ += free_space;

        bool retrier_active = retrier_->is_active();
        if (stopped_ or generate_active_ or retrier_active) {
            #ifdef PIPELINE_DEBUG
            YLOG_G(debug) << "skipping generate: stopped=" << stopped_ << " generate_active=" << generate_active_
                << " retrier_active=" << retrier_active;
            #endif
            return;
        }

        if (available_free_space_ < min_free_space_) {
            #ifdef PIPELINE_DEBUG
            YLOG_G(debug) << "skipping generate: too few free space available free_space=" << available_free_space_;
            #endif
            return;
        }

        generate_active_ = true;
        size_t available_free_space = available_free_space_;
        guard.unlock();
        try {
            generate(available_free_space);
        } catch (const std::exception& ex) {
            YLOG_G(error) << "unhandled error on generate: what=\"" << ex.what() << "\"";
        } catch (...) {
            YLOG_G(error) << "unknown unhandled error on generate";
        }
    }

    /**
    * Generator::on_generated or generate_failed should be used instead of direct output->put_range.
    */
    virtual void generate(std::size_t free_space) = 0;

    void reset_retry_intervals()
    {
        scoped_lock guard(mutex_);
        retrier_->reset();
    }

    void generate_failed()
    {
        scoped_lock guard(mutex_);
        assert(generate_active_);
        generate_active_ = false;
        if (!stopped_) {
            retrier_->next_retry();
        }
    }

    template <typename Collection>
    void on_generated(std::shared_ptr<Collection> data, bool retry_later = false)
    {
        scoped_lock guard(mutex_);
        assert(generate_active_);

        std::size_t generated_count = data->size();
        if (generated_count > available_free_space_) {
            YLOG_G(error) << "[base_generator::put_range] generated size > available free space, some data will be lost:"
                " generated=" << generated_count <<
                " free_space=" << available_free_space_;
        }
        if (not data->empty()) {
            #ifdef PIPELINE_DEBUG
            YLOG_G(debug) << "[base_generator::put_range] size=" << data->size();
            #endif
            if (auto stream = output_) {
                guard.unlock();
                stream->put_range(data);
                guard.lock();
            }
        }
        available_free_space_ -= (generated_count > available_free_space_ ? available_free_space_ : generated_count);
        generate_active_ = false;
        if (stopped_) return;

        retrier_->reset();
        if (generated_count == 0 || retry_later) {
            retrier_->next_retry();
        }
        else if (available_free_space_ > 0) {
            // @todo: why not retrier::restart?
            on_free_space_impl(guard);
        }
    }

    // is_retry_started and is_last_retry_interval are virtual for test purposes
    virtual bool is_retry_started() const
    {
        return retrier_->is_retry_started();
    }

    virtual bool is_last_retry_interval() const
    {
        return retrier_->is_last_retry_interval();
    }

private:
    stream_ptr output_;
    const std::size_t min_free_space_;
    bool stopped_;
    bool generate_active_;
    retrier_ptr retrier_;
    std::size_t available_free_space_;
    mutex mutex_;
};

}
