#pragma once

#include <list>

#include "generator.h"
#include "processor.h"
#include "commit_listener.h"

namespace pipeline {

template<typename StreamT>
class Plan {
public:
    typedef std::shared_ptr<Generator<StreamT>> generator_ptr;
    typedef std::shared_ptr<Processor<StreamT>> processor_ptr;
    typedef std::shared_ptr<CommitListener<StreamT>> listener_ptr;

    Plan(const Plan& other)
    : generator_(other.generator_), processors_(other.processors_),
      listener_(other.listener_), stopped_(other.stopped_ ? true : false)
    {}

    Plan(generator_ptr generator)
    : generator_(generator), stopped_(true)
    {}

    Plan& operator|(processor_ptr processor)
    {
        assert(generator_);
        if (processors_.empty()) {
            generator_->output(processor->input());
        } else {
            processors_.back()->output(processor->input());
        }
        processors_.push_back(processor);
        return *this;
    }

    Plan& operator|(listener_ptr listener)
    {
        assert(not listener_);
        assert(processors_.size());
        listener->input(processors_.back()->input());
        listener_ = listener;
        return *this;
    }

    bool is_stopped() const
    {
        return stopped_;
    }

    void start()
    {
        if (!stopped_) return;
        stopped_ = false;

        if (listener_) {
            listener_->start();
        }
        for (auto proc = processors_.rbegin(); proc != processors_.rend(); proc++) {
            (*proc)->start();
        }
        if (generator_) {
            generator_->start();
        }
    }

    void stop()
    {
        stopped_ = true;

        if (generator_) {
            generator_->stop();
        }
        for (auto& proc : processors_) {
            proc->stop();
        }
        if (listener_) {
            listener_->stop();
        }
    }

    void clear()
    {
        generator_.reset();
        processors_.clear();
        listener_.reset();
    }

    const generator_ptr generator() const
    {
        return generator_;
    }

    const std::list<processor_ptr> processors() const
    {
        return processors_;
    }

private:
    generator_ptr generator_;
    std::list<processor_ptr> processors_;
    listener_ptr listener_;
    std::atomic_bool stopped_;
};

template <typename StreamT>
inline Plan<StreamT> from(typename Plan<StreamT>::generator_ptr generator)
{
    return Plan<StreamT>(generator);
}

}
