#pragma once

#include <pipeline/stream_settings.h>
#include <pipeline/types.h>
#include <boost/bind.hpp>
#include <boost/asio.hpp>
#include <string>

namespace pipeline {

template <typename StreamT>
class Processor : public std::enable_shared_from_this<Processor<StreamT>>
{
protected:
    typedef StreamT stream_t;
    typedef typename stream_t::this_ptr stream_ptr;

public:
    Processor(boost::asio::io_service& io, const StreamSettings& stream_settings)
    : input_(std::make_shared<stream_t>(io, stream_settings))
    {}

    virtual ~Processor()
    {}

    stream_ptr input()
    {
        scoped_lock guard(mutex_);
        return input_;
    }

    void output(stream_ptr out)
    {
        auto stream = input();
        if (stream) stream->pipe(out);
    }

    virtual void start()
    {
        auto stream = input();
        if (stream) stream->reset_data_handler(boost::bind(&Processor::on_data, this->shared_from_this(), _1, _2, _3));
    }

    virtual void stop()
    {
        scoped_lock guard(mutex_);
        auto stream = input_;
        input_.reset();
        guard.unlock();

        if (stream)
            stream->stop();
    }

    virtual std::string name() const
    {
        return typeid(*this).name();
    }

protected:
    virtual void on_data(stream_ptr stream, std::size_t begin_id, std::size_t end_id) = 0;

private:
    mutex mutex_;
    stream_ptr input_;
};

}
