#pragma once

#include <mutex>

#include <yplatform/time_traits.h>
#include <boost/function.hpp>
#include <boost/bind.hpp>
#include <boost/asio.hpp>

namespace pipeline {

using std::string;
namespace time_traits = yplatform::time_traits;

class ProgressiveRetrier : public std::enable_shared_from_this<ProgressiveRetrier>
{
public:
    typedef boost::function<void(void)> callback_t;
    typedef time_traits::duration interval_t;
    typedef std::vector<interval_t> intervals_t;

    ProgressiveRetrier(boost::asio::io_service& io, intervals_t intervals)
    : retry_active_(false),
      timer_(io),
      intervals_(intervals)
    {
        if (intervals.empty()) intervals_ = { time_traits::seconds(2) };
        current_interval_ = intervals_.cbegin();
    }

    bool is_retry_started() const
    {
        scoped_lock guard(mutex_);
        return current_interval_ != intervals_.cbegin();
    }

    bool is_last_retry_interval() const
    {
        scoped_lock guard(mutex_);
        return current_interval_ + 1 == intervals_.cend();
    }

    bool is_active() const
    {
        scoped_lock guard(mutex_);
        return retry_active_;
    }

    void start(const callback_t& callback)
    {
        scoped_lock guard(mutex_);
        callback_ = callback;
        reset_impl();
    }

    void stop()
    {
        scoped_lock guard(mutex_);
        callback_.clear();
        reset_impl();
    }

    void reset()
    {
        scoped_lock guard(mutex_);
        reset_impl();
    }

    void next_retry()
    {
        scoped_lock guard(mutex_);
        if (retry_active_ || !callback_) return;

        next_retry_attempt();
    }

private:
    void reset_impl()
    {
        timer_.cancel();
        retry_active_ = false;
        current_interval_ = intervals_.cbegin();
    }

    void next_retry_attempt()
    {
        bool is_last_interval = current_interval_ + 1 == intervals_.end();

        retry_active_ = true;
        auto interval = is_last_interval ? current_interval_ : current_interval_++;
        auto cb = boost::bind(&ProgressiveRetrier::on_retry, this->shared_from_this(),
                              boost::asio::placeholders::error);
        timer_.expires_from_now(*interval);
        timer_.async_wait(cb);
    }

    void on_retry(const boost::system::error_code& err)
    {
        if (err == boost::asio::error::operation_aborted) {
            return;
        }
        scoped_lock guard(mutex_);
        retry_active_ = false;
        auto cb = callback_;
        guard.unlock();

        if (cb) cb();
    }

private:
    mutable mutex mutex_;

    bool retry_active_;
    time_traits::timer timer_;
    intervals_t intervals_;
    intervals_t::const_iterator current_interval_;
    callback_t callback_;
};

}
