#pragma once

#include <pipeline/types.h>
#include <mutex>
#include <functional>

#include <boost/bind.hpp>
#include <boost/asio.hpp>

#include "base_stream.h"
#include "profiler.h"

namespace pipeline {

#define STREAM_PROFILER(name) SIMPLE_RAII_PROFILER(label_, #name);

template<typename Data>
class StreamStrand: public std::enable_shared_from_this<StreamStrand<Data>>
{
    typedef BaseStream<Data> implementation_t;
    typedef StreamStrand<Data> this_t;
public:
    typedef typename std::shared_ptr<this_t> this_ptr;

    typedef typename implementation_t::temp_collection_ptr temp_collection_ptr;
    typedef typename implementation_t::Event Event;

    typedef boost::function<void (this_ptr, std::size_t, std::size_t)> on_data_callback_t;
    typedef boost::function<void (temp_collection_ptr)> on_commit_callback_t;
    typedef boost::function<void (std::size_t)> on_free_space_callback_t;

    typedef boost::asio::io_service::strand strand_t;
    typedef std::shared_ptr<strand_t> strand_ptr;

    StreamStrand(boost::asio::io_service& io, const StreamSettings& settings)
    : impl_(settings),
      consume_queued_(false),
      commit_queued_(false),
      strand_(std::make_shared<strand_t>(io)),
      stopped_(false)
    {}

    template <template <typename...> class Collection>
    void put_range(std::shared_ptr<Collection<Data>> data) noexcept
    {
        STREAM_PROFILER(put_range)
        strand_->post(boost::bind(&this_t::put_range_impl<Collection>, this->shared_from_this(), data));
    }

    template <template <typename...> class Collection, typename OverflowHandler>
    void put_range(std::shared_ptr<Collection<Data>> data, OverflowHandler&& handler) noexcept
    {
        STREAM_PROFILER(put_range)
        strand_->post(boost::bind(&this_t::put_range_checked_impl<Collection,OverflowHandler>,
            this->shared_from_this(), data, std::forward<OverflowHandler>(handler)));
    }

    void commit(std::size_t id) noexcept
    {
        STREAM_PROFILER(commit)
        strand_->post(boost::bind(&this_t::commit_impl, this->shared_from_this(), id));
    }

    void commit_until(std::size_t id) noexcept
    {
        STREAM_PROFILER(commit_until)
        strand_->post(boost::bind(&this_t::commit_until_impl, this->shared_from_this(), id));
    }

    /**
    * at() is not thread-safe. Should be called only from on_data event handler.
    */
    const Data& at(std::size_t id) const
    {
        STREAM_PROFILER(at)
        assert(strand_->running_in_this_thread());
        return impl_[id];
    }

    /**
     * Setters
     */
    void label(const std::string& name) { label_ = name; }
    const std::string& label() const { return label_; }

    void set_min_grouped_consume_size(std::size_t count = 0)
    {
        STREAM_PROFILER(set_min_grouped_consume_size)
        impl_.set_min_grouped_consume_size(count);
    }

    /**
     * Start/Stop functions
     * handlers are supposed to be set on start, for stopping call stop()
     */
    void pipe(this_ptr output) noexcept
    {
        STREAM_PROFILER(pipe)
        reset_commit_handler(
            boost::bind(&this_t::put_range<std::vector>, output, _1), true
        );
        output->reset_free_space_handler(strand_->wrap(
            boost::bind(&this_t::on_free_space_for_consume_impl, this->shared_from_this(), _1)
        ));
    }

    void reset_data_handler(const on_data_callback_t& cb = on_data_callback_t())
    {
        STREAM_PROFILER(reset_data_handler)
        strand_->dispatch(boost::bind(&this_t::reset_data_handler_impl, this->shared_from_this(), cb));
    }

    void reset_commit_handler(const on_commit_callback_t& cb = on_commit_callback_t(), bool is_output_fixed_size = false)
    {
        STREAM_PROFILER(reset_commit_handler)
        strand_->dispatch(boost::bind(&this_t::reset_commit_handler_impl, this->shared_from_this(), cb, is_output_fixed_size));
    }

    void reset_free_space_handler(const on_free_space_callback_t& cb = on_free_space_callback_t())
    {
        STREAM_PROFILER(reset_free_space_handler)
        strand_->dispatch(boost::bind(&this_t::reset_free_space_handler_impl, this->shared_from_this(), cb));
    }

    void stop()
    {
        STREAM_PROFILER(stop)
        scoped_lock guard(mutex_);
        if (stopped_) return;
        stopped_ = true;

        on_data_.clear();
        on_commit_.clear();
        on_free_space_.clear();
        impl_.reset_consume_size_observer(false);
    }

    /**
     * Getters (stats)
     */
    std::size_t buffer_size() const { return impl_.buffer_size(); }
    std::size_t total_commited_size() const { return impl_.total_commited_size(); }
    std::size_t begin_id() const { return impl_.begin_id(); }

protected:
    template <template <typename...> class Collection>
    void put_range_impl(std::shared_ptr<Collection<Data>> data)
    {
        STREAM_PROFILER(put_range_impl)
        if (is_stopped()) return;
        impl_.put_range(data);
        process_next_event();
    }

    template <template <typename...> class Collection, typename OverflowHandler>
    void put_range_checked_impl(std::shared_ptr<Collection<Data>> data,
        OverflowHandler& handler)
    {
        STREAM_PROFILER(put_range_impl)
        if (is_stopped()) return;
        if (data->size() > impl_.free_space()) {
          notify_overflow(std::move(data), std::move(handler));
          return;
        }
        impl_.put_range(data);
        process_next_event();
    }

    void commit_impl(std::size_t id)
    {
        STREAM_PROFILER(commit_impl)
        if (is_stopped()) return;
        commit_ids.push_back(id);
        if (!commit_queued_) {
            commit_queued_ = true;
            strand_->post(boost::bind(&this_t::commit_all, this->shared_from_this()));
        }
    }

    void commit_all()
    {
        STREAM_PROFILER(commit_all)
        if (is_stopped()) return;
        impl_.commit_all(commit_ids);
        commit_ids.clear();
        commit_queued_ = false;
        process_next_event();
    }

    void commit_until_impl(std::size_t id)
    {
        STREAM_PROFILER(commit_until_impl)
        if (is_stopped()) return;
        impl_.commit_until(id);
        process_next_event();
    }

    void on_free_space_for_consume_impl(std::size_t free_space)
    {
        STREAM_PROFILER(on_free_space_for_consume_impl)
        if (is_stopped()) return;
        impl_.on_free_space_for_consume(free_space);
        process_next_event();
    }

    bool is_stopped()
    {
        STREAM_PROFILER(is_stopped)
        scoped_lock guard(mutex_);
        return stopped_;
    }

    inline void process_next_event()
    {
        STREAM_PROFILER(process_next_event)
        switch(impl_.next_event()) {
            case Event::NewDataReceived:
                notify_on_data(impl_.next_interval());
                break;
            case Event::ReadyForConsume:
                enqueue_consume();
                break;
            case Event::None:
                break;
        }
    }

    void enqueue_consume()
    {
        STREAM_PROFILER(enqueue_consume)
        if (consume_queued_) return;
        consume_queued_ = true;
        // post and consume_queued_ are used to merge several dispatched consume operations into one
        // (in case when processor uses only commit(id) function)
        strand_->post(boost::bind(&this_t::transfer_consumed_data, this->shared_from_this()));
    }

    void transfer_consumed_data()
    {
        STREAM_PROFILER(transfer_consumed_data)
        consume_queued_ = false;
        while (auto splice = impl_.consume()) {
            notify_on_commit(splice);
            notify_on_free_space(splice->size());
        }
        process_next_event();
    }

    void reset_data_handler_impl(const on_data_callback_t& cb)
    {
        STREAM_PROFILER(reset_data_handler_impl)
        scoped_lock guard(mutex_);
        if (stopped_) return;
        on_data_ = cb;
    }

    void reset_commit_handler_impl(const on_commit_callback_t& cb, bool is_fixed_size)
    {
        STREAM_PROFILER(reset_commit_handler_impl)
        scoped_lock guard(mutex_);
        if (stopped_) return;
        on_commit_ = cb;
        guard.unlock();

        impl_.reset_consume_size_observer(is_fixed_size);
    }

    void reset_free_space_handler_impl(const on_free_space_callback_t& cb)
    {
        STREAM_PROFILER(reset_free_space_handler_impl)
        scoped_lock guard(mutex_);
        if (stopped_) return;
        on_free_space_ = cb;
        guard.unlock();

        notify_on_free_space(impl_.free_space());
    }

    void notify_on_data(std::pair<std::size_t, std::size_t> interval)
    {
        STREAM_PROFILER(notify_on_data)
        std::size_t begin_id = interval.first;
        std::size_t end_id = interval.second;

        #ifdef PIPELINE_DEBUG
        YLOG_G(debug) << "[" << label_ << "].notify_on_data begin_id=" << begin_id << " end_id=" << end_id;
        #endif
        assert(end_id > begin_id);

        scoped_lock guard(mutex_);
        if (auto cb = on_data_) {
            guard.unlock();
            wrap_callback("on_data", cb, this->shared_from_this(), begin_id, end_id);
        }
    }

    void notify_on_commit(temp_collection_ptr data)
    {
        STREAM_PROFILER(notify_on_commit)
        scoped_lock guard(mutex_);
        if (auto cb = on_commit_) {
            guard.unlock();
            wrap_callback("on_commit", cb, data);
        }
    }

    void notify_on_free_space(std::size_t free_space)
    {
        STREAM_PROFILER(notify_on_free_space)
        scoped_lock guard(mutex_);
        if (auto cb = on_free_space_) {
            guard.unlock();
            wrap_callback("on_free_space", cb, free_space);
        }
    }

    template <typename Handler, typename Collection>
    void notify_overflow(Handler&& handler, Collection&& data)
    {
        STREAM_PROFILER(notify_overflow)
        wrap_callback("notify_overflow", std::forward<Collection>(data),
            std::forward<Handler>(handler));
    }

    template <typename Callback, typename... Args>
    void wrap_callback(const char* alias, const Callback& cb, Args&&... args) {
        STREAM_PROFILER(wrap_callback)
        assert_strand();

        try {
            cb(std::forward<Args>(args)...);
        } catch (const std::exception& ex) {
            YLOG_G(error) << "[" << label_ << "].wrap_callback error while executing: callback=\"" << alias << "\""
                " what=\"" << ex.what() << "\"";
            assert(false && "Unexpected error from callback");
        } catch (...) {
            YLOG_G(error) << "[" << label_ << "].wrap_callback unknown error while executing: callback=\"" << alias << "\"";
            assert(false && "Unexpected error from callback");
        }
    }

protected:
    void assert_strand() const
    {
        assert(strand_->running_in_this_thread());
    }

private:
    on_data_callback_t on_data_;
    on_commit_callback_t on_commit_;
    on_free_space_callback_t on_free_space_;

    implementation_t impl_;
    std::string label_;
    bool consume_queued_;

    bool commit_queued_;
    std::vector<size_t> commit_ids;

    strand_ptr strand_;
    mutex mutex_;
    bool stopped_;
};

#undef STREAM_PROFILER

}
