#pragma once

#include <utility>
#include <cassert>

namespace pipeline {

using std::size_t;

typedef std::pair<size_t, size_t> interval_t;

class Window
{
    size_t begin_id_;
    size_t prev_end_id_;
    size_t end_id_;
    size_t max_size_;

public:
    Window(size_t id, size_t max_size)
    : begin_id_(id),
      prev_end_id_(id),
      end_id_(id),
      max_size_(max_size)
    {}

    void begin_id(size_t id)
    {
        assert(id >= begin_id_);
        begin_id_ = id;
        if (begin_id_ > end_id_) {
            prev_end_id_ = end_id_ = begin_id_;
        } else if (begin_id_ > prev_end_id_) {
            prev_end_id_ = begin_id_;
        }
    }

    size_t begin_id() const
    {
        return begin_id_;
    }

    void end_id(size_t new_end_id, size_t exclude_count = 0)
    {
        assert(end_id_ <= new_end_id);
        if (end_id_ == new_end_id) return;

        size_t full_size = total_size();
        assert(exclude_count <= full_size);
        size_t actual_size = full_size - exclude_count;

        if (actual_size < max_size_) {
            size_t increase_count = std::min(max_size_ - actual_size, new_end_id - end_id_);
            end_id_ += increase_count;
        }
    }

    size_t end_id() const
    {
        return end_id_;
    }

    size_t total_size() const
    {
        return end_id_ - begin_id_;
    }

    bool next_interval_available() const
    {
        return prev_end_id_ < end_id_;
    }

    interval_t next_interval()
    {
        size_t interval_begin = prev_end_id_;
        prev_end_id_ = end_id_;
        return std::make_pair(interval_begin, end_id_);
    }
};

}
