#pragma once

#include <yxiva/core/batch_key.h>
#include <yxiva/core/message.h>
#include <yxiva/core/packing.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <algorithm>

namespace yxiva {
namespace reaper {

struct hub_request_factory
{
  hub_request_factory(const unsubscribe_task_ptr& task)
  : settings_(*task->settings)
  , task_(*task)
  {}

  // Prepare /batch_list_json request.
  yhttp::request list()
  {
    using ymod_httpclient::url_encode;

    string url = settings_.list_url + url_encode({{"uid", task_.event.uid}});
    string body = url_encode({{"services", boost::algorithm::join(*task_.services, ",")}}, '\0');

    return yhttp::request::POST(url, std::move(body));
  }

  // Prepare /batch_binary_notify request
  // for a given service and recipients.
  yhttp::request notify(size_t service_index)
  {
    auto& unsubscribe_list = task_.service_tasks[service_index].unsubscribe_list;
    auto& service = task_.services->at(service_index);

    message msg;
    msg.uid = task_.event.uid;
    msg.service = service;
    msg.event_ts = std::time(nullptr);
    msg.operation = "unsubscribe";
    msg.bright = true;
    msg.raw_data = R"({"event":"unsubscribe","reason":")"
      + task_.event.name + "\"}";
    msg.transit_id = task_.ctx->uniq_id();
    msg.set_flag(message_flags::ignore_filters);

    batch_keys keys;
    keys.reserve(unsubscribe_list.size());
    for (auto it = unsubscribe_list.begin();
         it != unsubscribe_list.end() && it->notify;
         ++it) {
      keys.push_back({task_.event.uid, it->id});
    }

    string url = settings_.batch_binary_notify_url + ymod_httpclient::url_encode(
      {{"service", service},
       {"size", keys.size()},
       {"agent", "reaper"}});

    string body = pack(msg);
    body += pack(keys);
    return yhttp::request::POST(url, std::move(body));
  }

  // Prepare /unsubscribe request for a given service and subscription id.
  yhttp::request unsubscribe(size_t service_index)
  {
    using ymod_httpclient::url_encode;
    using boost::algorithm::join;
    using boost::adaptors::transformed;
    auto& unsubscribe_list = task_.service_tasks[service_index].unsubscribe_list;

    string url = settings_.unsubscribe_url + url_encode(
      {{"service", task_.services->at(service_index)},
       {"uid", task_.event.uid},
       {"ts", task_.event.time}});

    string body = url_encode({{"ids", join(unsubscribe_list
        | transformed([] (const sub_data& sub) { return sub.id; }), ",")}}, '\0');

    return yhttp::request::POST(url, std::move(body));
  }

  const settings& settings_;
  unsubscribe_task& task_;
};

}}
