#pragma once

#include "settings.h"
#include "unsubscribe_task.h"
#include "unsubscribe_coro.h"

#include <yxiva/core/conf.h>
#include <yxiva/core/message.h>
#include <yxiva/core/packing.hpp>
#include <ymod_xconf/local_conf_storage.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/spinlock.h>
#include <ymod_httpclient/call.h>
#include <ymod_webserver/server.h>
#include <algorithm>
#include <functional>
#include <memory>

namespace yxiva {
namespace reaper {

class mod: public yplatform::module
{
  using sync_t = yplatform::spinlock;
  using scoped_lock = std::lock_guard<sync_t>;
  using services_type = std::vector<string>;
public:
  void init(const yplatform::ptree& config)
  {
    http_client_ = yplatform::find<yhttp::call,
      std::shared_ptr>("http_client");
    update_strand_ = std::make_shared<boost::asio::io_service::strand>(
      *yplatform::find_reactor(config.get("reactor", "global"))->io());
    auto web_server = yplatform::find<ymod_webserver::server>("web_server");

    web_server->bind("default", {"/passport_hook"}, std::bind(
      &mod::process_callback, shared_from(this), std::placeholders::_1));
    web_server->bind("default", {"/ping"}, [] (ymod_webserver::http::stream_ptr stream) {
      stream->result(ymod_webserver::codes::ok, "pong");
    });

    settings_ = std::make_shared<settings>();
    settings_->read_config(config);
    services_ = settings_->use_configured_service_list
      ? read_services(config)
      : std::make_shared<services_type>();

    if (!settings_->use_configured_service_list) {
      // Use xconf to keep services list up-to-date.
      auto xconf = yplatform::find<ymod_xconf::local_conf_storage>("xconf");
      xconf->subscribe_updates(ymod_xconf::config_type::SERVICE, update_strand_->wrap(
        std::bind(&mod::update_services, shared_from(this), std::placeholders::_1)));
    }
  }

  void reload (const yplatform::ptree& config)
  {
    auto new_settings = std::make_shared<settings>();
    new_settings->read_config(config);
    auto new_services = new_settings->use_configured_service_list
      ? read_services(config)
      : nullptr;

    scoped_lock lock(sync_);
    // Store old and new settings in conveniently named variables for further use.
    auto old_settings = settings_;
    settings_ = new_settings;
    // What to do with use config list (disable xconf) state in old and new config:
    // old / new / action
    //  -     -    leave old service list (managed by xconf)
    //  -     +    fallback mode activated, overwrite services
    //  +     -    error - disabling and re-enabling xconf may lead to data loss
    //             because xconf module doesn't support un- and resubscribing
    //  +     +    using configured list, overwrite with new config settings.
    if (new_settings->use_configured_service_list) {
      // Table case "old * new +", overwrite services for any "old" value.
      services_.swap(new_services);
    } else if (old_settings->use_configured_service_list) {
      // Table case "old + new -", report error and leave service_list as is.
      YLOG_L(error) << "failed to load new services: can't enable xconf";
      new_settings->use_configured_service_list = true;
    }
    // Table case "old - new -", do nothing.
  }

private:
  void process_callback(ymod_webserver::http::stream_ptr stream)
  {
    auto settings = get_settings();
    auto services = get_services();
    auto mod_log = yplatform::find<::yxiva::reaper::mod_log, std::shared_ptr>("mod_log");
    // no services to process
    if (services->empty()) {
      WEB_RESPONSE_LOG_L(info, stream, ok, "no services to process");
      return;
    }
    message msg;
    try {
      auto req = stream->request();
      unpack(string(req->raw_body.begin(), req->raw_body.end()), msg);
    } catch (const std::exception& e) {
      WEB_RESPONSE_LOG_L(info, stream, bad_request, "failed to unpack body");
      mod_log->event_dropped(stream->ctx(), "failed to unpack");
      return;
    }
    // Log passport event because it doesn't appear in the access log.
    mod_log->event_received(stream->ctx(), msg);
    json_value event_data;
    auto parse_res = json_parse(event_data, msg.raw_data);
    if (!parse_res)
    {
      WEB_RESPONSE_LOG_L(info, stream, bad_request, parse_res.error_reason);
      mod_log->event_dropped(stream->ctx(), "failed to parse json");
      return;
    }
    unsubscribe_task_ptr task;
    auto load_res = create_task(event_data, task,
      stream, http_client_, settings, services, mod_log);
    if (!load_res)
    {
      WEB_RESPONSE_LOG_L(info, stream, bad_request,
        "bad event: " + load_res.error_reason);
      mod_log->event_dropped(stream->ctx(), load_res.error_reason);
      return;
    }
    if (task->event.type == event_type::not_supported) {
      auto error = "unsupported event: " + task->event.name;
      WEB_RESPONSE_LOG_L(info, stream, accepted, error);
      mod_log->event_dropped(stream->ctx(), error);
      return;
    }
    mod_log->event_start(stream->ctx(), task->event.name,
      task->event.uid, task->event.connection_id);
    unsubscribe_coro coro(task);
    coro();
  }

  settings_ptr get_settings()
  {
    scoped_lock lock(sync_);
    return settings_;
  }

  std::shared_ptr<services_type> get_services()
  {
    scoped_lock lock(sync_);
    return services_;
  }

  void update_services(ymod_xconf::conf_list_ptr confs)
  {
    if (get_settings()->use_configured_service_list) {
      YLOG_L(info) << "services won't be updated: switched to fallback list";
      return;
    }
    auto new_services = std::make_shared<services_type>(*get_services());
    for (auto& item: confs->items) {
      try {
        service_properties data;
        unpack(item.configuration, data);
        if (data.is_passport) {
          // Keep service list unique.
          auto it = std::lower_bound(new_services->begin(), new_services->end(), data.name);
          if (it == new_services->end() || *it != data.name) {
            YLOG_L(info) << "added service " << data.name;
            new_services->insert(it, data.name);
          }
        }
      } catch (const std::exception& ex) {
        YLOG_L(error) << "failed to unpack item " << item.name << ": " << ex.what();
      }
    }
    // Additional check in case of reload().
    scoped_lock lock(sync_);
    if (!settings_->use_configured_service_list) {
      services_.swap(new_services);
    }
  }

  std::shared_ptr<yhttp::call> http_client_;
  std::shared_ptr<boost::asio::io_service::strand> update_strand_;
  settings_ptr settings_;
  std::shared_ptr<services_type> services_;
  sync_t sync_;
};

}}
