#pragma once

#include <yxiva/core/message.h>
#include <yplatform/module.h>
#include <yplatform/log/typed.h>
#include <yplatform/task_context.h>
#include <yplatform/find.h>

#define make_context_attr(ctx) t::make_attr("context", ctx ? ctx->uniq_id() : "")

namespace yxiva {
namespace reaper {

namespace t = yplatform::log::typed;

class mod_log: public yplatform::module
{
public:
    void init(const yplatform::ptree& config)
    {
        logger_ = yplatform::find<yplatform::log::typed::logger>(
          config.get("typed_log_id", "tskv"));
    }

    const char* bool_message(bool val)
    {
        return val ? "yes" : "no";
    }

    void event_received(task_context_ptr ctx, const message& msg)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", "received")
          << t::make_attr("transit_id", msg.transit_id) << t::make_attr("raw_event", msg.raw_data);
    }

    void event_dropped(task_context_ptr ctx, const string& reason)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", "dropped")
          << t::make_attr("error", reason);
    }

    void event_start(task_context_ptr ctx, const string& event,
      const string& uid, const string& connection_id)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", "start")
          << t::make_attr("event", event) << t::make_attr("uid", uid)
          << t::make_attr("connection_id", connection_id);
    }

    void list_finished(task_context_ptr ctx, size_t sub_count)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", "listed")
          << t::make_attr("subscription_count", sub_count);
    }

    void subscription_action(task_context_ptr ctx, const string& service,
      const string& id, bool unsubscribe, bool notify, const string& connection_id)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", "subscription")
          << t::make_attr("service", service) << t::make_attr("id", id)
          << t::make_attr("unsubscribe_required", bool_message(unsubscribe))
          << t::make_attr("notify_required", bool_message(notify))
          << t::make_attr("connection_id", connection_id);
    }

    void action_error(task_context_ptr ctx, const string& action, const string& reason)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", action + " error")
          << t::make_attr("error", reason);
    }

    void notify_finished(task_context_ptr ctx, const string& service,
      const string& id, bool unsubscribed)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", "notified")
          << t::make_attr("service", service) << t::make_attr("id", id)
          << t::make_attr("unsubscribed", bool_message(unsubscribed));
    }

    void unsubscribe_finished(task_context_ptr ctx, const string& service, const string& id)
    {
        YLOG((*logger_), info) << make_context_attr(ctx) << t::make_attr("status", "unsubscribed")
          << t::make_attr("service", service) << t::make_attr("id", id);
    }

    void event_end(task_context_ptr ctx, bool success)
    {
        YLOG((*logger_), info) << make_context_attr(ctx)
          << t::make_attr("status", success ? "finished" : "failed");
    }

private:
    boost::optional<yplatform::log::typed::logger> logger_;
};

}}

#undef make_context_attr
#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::reaper::mod_log)
