#pragma once

#include "extractor.h"

#include <yxiva/core/callbacks.h>
#include <yxiva/core/notify_status.h>
#include <yplatform/util/sstream.h>
#include <algorithm>

namespace yxiva {
namespace reaper {

struct response_processor
{
  response_processor(const unsubscribe_task_ptr& task)
  : settings_(*task->settings)
  , task_(*task)
  {}

  bool unsubscribe_required(const json_value& sub)
  {
    auto& event = task_.event;
    if (event.time >= get_field(sub, field_codes::sub_init_time)) {
      switch (event.type) {
        case event_type::account_invalidate:
          return true;
        case event_type::session_invalidate: {
          auto conn_id = get_field(sub, field_codes::sub_connection_id);
          return conn_id.size() && (event.connection_id == conn_id);
        }
        default: ;
      }
    }
    return false;
  }

  bool last_push_required(const json_value& sub)
  {
    return !callback_uri::is_mobile_uri(get_field(sub, field_codes::sub_url));
  }

  operation::result list(const string& list_response)
  {
    json_value sub_list;
    auto res = json_parse(sub_list, list_response, json_type::tarray);
    if (res) {
      // Most of /list_json responses appear to be empty lists and logging them
      // is excessive, so we only do debug-level loggigng of non-empty lists
      if (sub_list.size()) {
        YLOG_CTX_GLOBAL(task_.ctx, debug) << "uid = " << task_.event.uid
          << ", hub list response: \"" << list_response << "\"";
      }

      for (auto&& sub: sub_list.array_items()) {
        if (!has_field(sub, field_codes::sub_id)
          || !has_field(sub, field_codes::sub_service)
          || !has_field(sub, field_codes::sub_url)
          || !has_field(sub, field_codes::sub_init_time)) {
          return "malformed subscription " + json_write(sub);
        }

        string service = get_field(sub, field_codes::sub_service);
        auto it = std::lower_bound(task_.services->begin(), task_.services->end(), service);
        if (it == task_.services->end() || *it != service) {
          YLOG_CTX_GLOBAL(task_.ctx, warning) << "list unknown service " << service;
          continue;
        }
        auto service_index = std::distance(task_.services->begin(), it);
        auto& unsubscribe_list = task_.service_tasks[service_index].unsubscribe_list;
        string subscription_id = get_field(sub, field_codes::sub_id);
        bool unsubscribe = unsubscribe_required(sub);
        bool notify = unsubscribe && last_push_required(sub);

        if (unsubscribe) {
          unsubscribe_list.push_back({subscription_id, notify, true});
        }
        task_.mod_log->subscription_action(task_.ctx, service, subscription_id,
          unsubscribe, notify, get_field(sub, field_codes::sub_connection_id));
      }
      for (size_t i = 0; i < task_.services->size(); ++i) {
        auto& unsubscribe_list = task_.service_tasks[i].unsubscribe_list;
        // Count subscriptions to process in each service,
        // also put subscriptions with notifications in front to use
        // batch_binary_notify's response indices.
        task_.service_tasks[i].unsubscribe_count = unsubscribe_list.size();
        auto notify_end = std::partition(unsubscribe_list.begin(), unsubscribe_list.end(),
          [] (const sub_data& data) { return data.notify; });
        task_.service_tasks[i].notify_count =
          std::distance(unsubscribe_list.begin(), notify_end);
      }
      // Especially helpful with 0 subscription case.
      task_.mod_log->list_finished(task_.ctx, sub_list.size());
    } else {
      // Malformed json response should always be logged
      YLOG_CTX_GLOBAL(task_.ctx, error) << "uid = " << task_.event.uid
        << ", bad hub list response: \"" << list_response << "\", error:\""
        << res.error_reason << "\"";
    }
    return res;
  }

  operation::result notify(const string& body, size_t service_index)
  {
    auto& unsubscribe_list = task_.service_tasks[service_index].unsubscribe_list;
    auto& unsubscribe_count = task_.service_tasks[service_index].unsubscribe_count;
    auto& notify_count = task_.service_tasks[service_index].notify_count;

    string error_string;
    yplatform::sstream error_stream(error_string);
    std::vector<notify_status> results;
    try {
      unpack(body, results);
    } catch (const std::exception& e) {
      error_stream << "unpack error: " << e.what();
      return error_string;
    }
    for (auto& res: results) {
      switch (res.code) {
        case 200: // OK
          if (res.seq < unsubscribe_list.size()) {
            // Already notified – don't notify on retry.
            unsubscribe_list[res.seq].notify = false;
            --notify_count;
            task_.mod_log->notify_finished(task_.ctx, task_.services->at(service_index),
              res.subscription_id, false);
          }
          break;
        case 202: // Filtered (just in case)
        case 400: // Bad request (pointless to retry)
          if (res.seq < unsubscribe_list.size()) {
            // Meaningless to notify on retry.
            unsubscribe_list[res.seq].notify = false;
            --notify_count;
          }
          YLOG_CTX_GLOBAL(task_.ctx, info) << "uid = " << task_.event.uid
            << ", service = " << task_.services->at(service_index) << ", sub_id "
            << res.subscription_id << ": " << res.code << ", " << res.body;
          break;
        case 204: // No subscription (nothing to unsubscribe)
        case 205: // Unsubscribed by hub
          if (res.seq < unsubscribe_list.size()) {
            // Meaningless to unsubscribe or notify on retry.
            unsubscribe_list[res.seq].notify = false;
            unsubscribe_list[res.seq].unsubscribe = false;
            --unsubscribe_count;
            --notify_count;
          }
          task_.mod_log->notify_finished(task_.ctx, task_.services->at(service_index),
            res.subscription_id, true);
          break;
        default:
          error_stream << "sub_id " << res.subscription_id << " failed with code " << res.code
            << ", body \"" << res.body << "\")";
      }
    }
    if (error_string.size()) {
      // Rearrange subscriptions that still need to be notified on retry.
      std::partition(unsubscribe_list.begin(), unsubscribe_list.end(),
        [] (const sub_data& data) { return data.notify; });
      return error_string;
    }
    return operation::success;
  }

  const settings& settings_;
  unsubscribe_task& task_;
};

}}
