#pragma once

#include "unsubscribe_task.h"
#include "hub_request_factory.h"
#include "response_processor.h"

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace yxiva {
namespace reaper {

class unsubscribe_coro: public boost::asio::coroutine
{
public:
  unsubscribe_coro(unsubscribe_task_ptr task_)
  : task(std::move(task_))
  , ctx(task->ctx)
  , settings(task->settings)
  , service_index(0)
  , request(task)
  , process(task)
  {}

  // httpclient async_run handler
  void operator()(const boost::system::error_code& ec, yhttp::response response)
  {
    if (ec || response.status != 200) {
      log_error("reason: " + (ec
        ? ec.message()
        : "response code " + std::to_string(response.status)));
    } else {
      (*this)(&response);
    }
  }

  void operator()(yhttp::response* response = nullptr)
  {
    // Asio is_parent is weird and unreliable if more than one fork required.
    bool parent = false;
    // References for convenience.
    // Number of subscriptions left to process for each service.
    auto& unsubscribe_count = task->service_tasks[service_index].unsubscribe_count;
    // Number of subscriptions left to notify for each service.
    auto& notify_count = task->service_tasks[service_index].notify_count;
    // Subscription id-s to unsubscribe for each service.
    auto& unsubscribe_list = task->service_tasks[service_index].unsubscribe_list;

    if (ctx->is_cancelled()) {
      return;
    }

    try { reenter(this) {
      action.assign("list");
      // Call /batch_list_json for all services.
      yield task->http_client->async_run(ctx, request.list(), *this);

      // Limits scope of res to prevent cross initialization error.
      if (auto res = process.list(response->body)) {
      } else {
        log_error(res.error_reason);
        return;
      }

      // Fork for per-service processing.
      parent = true;
      while (parent && ++service_index < task->services->size()) {
        fork unsubscribe_coro(*this)();
      }
      if (parent) { service_index = 0; }

      // No subscriptions to unsubscribe, finished for current service.
      if (!unsubscribe_count) { yield break; }

      // Send last push if there are subscriptions that require it.
      if (notify_count) {
        action.assign("notify");

        // Retry client-side errors on notify and unsubscribe anyway.
        for (retry_counter = 0;
            retry_counter < settings->notify_max_attempts;
            ++retry_counter) {
          yield task->http_client->async_run(ctx, request.notify(service_index), *this);

          // Limits scope of res to prevent cross initialization error.
          if (auto res = process.notify(response->body, service_index)) {
            break;
          } else {
            log_error("reason: " + res.error_reason);
          }
        }
        if (retry_counter == settings->notify_max_attempts) {
          log_error("retry count is exceeded, proceed to unsubscribe regardless");
        }
      }

      // Unsubscribe, if all subscriptions weren't removed by hub after notify.
      if (unsubscribe_count) {
        action.assign("unsubscribe");
        yield task->http_client->async_run(ctx, request.unsubscribe(service_index), *this);
        // If reached this point, unsubscribe call was successful.
        for (auto& sub: unsubscribe_list) {
          if (sub.unsubscribe) {
            task->mod_log->unsubscribe_finished(ctx,
              task->services->at(service_index), sub.id);
          }
        }
      }
      // Coro won't be is_complete() for given service in case of failed unsubscribe.
    }} catch (const std::exception& e) {
      log_error(e.what());
      return;
    }

    if (is_complete()) {
      --task->service_count;
    }
  }

private:
  void log_error(const string& message)
  {
    task->mod_log->action_error(ctx, action, message);
  }

  unsubscribe_task_ptr task;
  task_context_ptr& ctx;
  settings_ptr& settings;
  size_t service_index;
  hub_request_factory request;
  response_processor process;
  // Indicates state of coro in verbose form for logging.
  string action;
  // Need to carry over yield.
  unsigned retry_counter;
};

#include <boost/asio/unyield.hpp>
}}
