#pragma once

#include "settings.h"
#include "extractor.h"
#include "mod_log.h"

#include <ymod_webserver/server.h>
#include <ymod_httpclient/call.h>
#include <yplatform/task_context.h>
#include <atomic>

namespace yxiva {
namespace reaper {

enum class event_type
{
  not_supported = 0,
  account_invalidate = 1,
  session_invalidate = 2
};

inline event_type event_type_from_name(const string& name)
{
  static const string event_account_invalidate = "account.invalidate";
  static const string event_session_invalidate = "session.invalidate";

  if (name == event_account_invalidate) {
    return event_type::account_invalidate;
  } else if (name == event_session_invalidate) {
    return event_type::session_invalidate;
  }
  return event_type::not_supported;
}

struct sub_data
{
  string id;        // Subscription id
  bool notify;      // Send last push before unsubscribe
  bool unsubscribe; // Need to unsubscribe (can change as a result of notify)
};

struct service_task
{
  // sub_data for each subscription of this service.
  std::vector<sub_data> unsubscribe_list;
  // Number of subscriptions left to unsubscribe for each service.
  size_t unsubscribe_count = 0;
  // Number of subscriptions left to notify for each service.
  size_t notify_count = 0;
};

struct unsubscribe_task
{
  unsubscribe_task(const ymod_webserver::http::stream_ptr& stream,
    const std::shared_ptr<yhttp::call>& http_client,
    const settings_ptr& settings,
    const std::shared_ptr<std::vector<string>>& services,
    const std::shared_ptr<::yxiva::reaper::mod_log>& mod_log)
  : unsubscribe_task(stream, stream->ctx(), http_client, settings, services, mod_log)
  {}

  // Convenience constructor with context arg for simpler test fixtures.
  unsubscribe_task(const ymod_webserver::http::stream_ptr& stream,
    const task_context_ptr& ctx,
    const std::shared_ptr<yhttp::call>& http_client,
    const settings_ptr& settings,
    const std::shared_ptr<std::vector<string>>& services,
    const std::shared_ptr<::yxiva::reaper::mod_log>& mod_log)
  : stream(stream)
  , ctx(ctx)
  , http_client(http_client)
  , settings(settings)
  , services(services)
  , service_count(services->size())
  , service_tasks(service_count)
  , mod_log(mod_log)
  {}

  ~unsubscribe_task()
  {
    bool success = stream && !service_count;
    if (success) {
      stream->result(ymod_webserver::codes::ok);
    }
    mod_log->event_end(ctx, success);
  }

  ymod_webserver::http::stream_ptr stream;
  task_context_ptr ctx;
  std::shared_ptr<yhttp::call> http_client;
  settings_ptr settings;
  std::shared_ptr<std::vector<string>> services;
  // Number of services left to process.
  std::atomic_size_t service_count;
  // Per-service task data.
  std::vector<service_task> service_tasks;
  std::shared_ptr<::yxiva::reaper::mod_log> mod_log;

  struct {
    string uid;
    string name;
    event_type type;
    string connection_id;
    uint64_t time;
  } event;
};

using unsubscribe_task_ptr = std::shared_ptr<unsubscribe_task>;

// Only creates task if the passed event is correct.
template <typename... Args>
inline operation::result create_task(const json_value& event_data,
  unsubscribe_task_ptr& task, Args&&... args)
{
  // Required fields
  if (!has_field(event_data, field_codes::event_uid)) return "missing uid field";
  if (!has_field(event_data, field_codes::event_name)) return "missing name field";
  if (!has_field(event_data, field_codes::event_timestamp)) {
    return "missing timestamp field";
  }

  auto name = get_field(event_data, field_codes::event_name);
  auto type = event_type_from_name(name);
  if (type == event_type::session_invalidate
    && !has_field(event_data, field_codes::event_connection_id)) {
    return "missing connection_id field";
  }

  task = std::make_shared<unsubscribe_task>(std::forward<Args>(args)...);

  task->event.name = name;
  task->event.type = type;
  task->event.uid = get_field(event_data, field_codes::event_uid);
  task->event.time = get_field(event_data, field_codes::event_timestamp);
  task->event.connection_id = get_field(event_data,
    field_codes::event_connection_id);

  return operation::success;
}

}}
