#include "impl.h"

#include <yxiva/core/gid.h>
#include <ymod_httpclient/util/url_parser.h>
#include <ymod_tvm/tvm.h>
#include <yplatform/find.h>
#include <random>

namespace yxiva {
namespace {
void bind_nodes(
    shared_ptr<ymod_tvm::interface> tvm,
    const string& service,
    const std::vector<string>& nodes)
{
    if (!tvm) return;
    for (auto& url : nodes)
    {
        string proto, host, uri;
        unsigned short port;
        if (auto err = yhttp::parse_url(url, proto, host, port, uri))
        {
            throw std::runtime_error(
                "can't parse node url \"" + url + "\": \"" +
                ymod_httpclient::http_error::message(err) + "\"");
        }
        tvm->bind_host(host, service);
    }
}
}

void hubrpc::settings::parse_ptree(const yplatform::ptree& conf)
{
    generate_nodes(conf);
    http.parse_ptree(conf.get_child("http"));
    if (auto tvm_node = conf.get_child_optional("tvm"))
    {
        tvm.module = tvm_node->get("module", string{});
        // Service name is required if using tvm.
        if (tvm.module.size())
        {
            tvm.service = tvm_node->get<string>("service");
        }
    }
    http.service_ticket_provider.module = tvm.module;
    for (auto i = 0UL; i < nodes_by_shard.size(); ++i)
    {
        if (nodes_by_shard[i].empty())
        {
            throw std::runtime_error("no nodes for shard " + std::to_string(i + 1));
        }
    }
}

void hubrpc::settings::generate_nodes(const yplatform::ptree& conf)
{
    nodes_by_shard.clear();

    if (conf.get_child_optional("nodes.list"))
    {
        auto num_shards = conf.get<unsigned>("nodes.list.num_shards");
        nodes_by_shard.resize(num_shards);
        for (auto i = 0U; i < num_shards; ++i)
        {
            auto key = "shard" + std::to_string(i + 1);
            for (auto& it :
                 boost::make_iterator_range(conf.get_child("nodes.list").equal_range(key)))
            {
                nodes_by_shard[i].push_back(it.second.data());
                L_(info) << "hub shard=" << i << " nodes+=" << it.second.data();
            }
        }
        return;
    }

    auto num_shards = conf.get<unsigned>("nodes.num_shards");
    auto nodes_per_shard = conf.get<unsigned>("nodes.nodes_per_shard");
    auto scheme = conf.get<string>("nodes.scheme");
    auto port = conf.get<unsigned>("nodes.port");
    auto shard_prefix = conf.get<string>("nodes.shard_prefix");
    auto fqdn_suffix = conf.get<string>("nodes.fqdn_suffix");

    nodes_by_shard.resize(num_shards);
    for (auto i = 0U; i < num_shards; ++i)
    {
        auto shard_name = shard_prefix + std::to_string(i + 1);
        nodes_by_shard[i].resize(nodes_per_shard);
        for (auto j = 0U; j < nodes_per_shard; ++j)
        {
            auto host_name = shard_name + "-" + std::to_string(j + 1);
            nodes_by_shard[i][j] = scheme + "://" + host_name + "." + shard_name + "." +
                fqdn_suffix + ":" + std::to_string(port);
            L_(info) << "hub shard=" << i << " nodes+=" << nodes_by_shard[i][j];
        }
    }
}

hubrpc::hubrpc(yplatform::reactor& reactor, const yplatform::ptree& conf)
{
    settings_.parse_ptree(conf);
    reactor_ = yplatform::reactor::make_not_owning_copy(reactor);
    auto tvm = settings_.tvm.module.size() ?
        yplatform::find<ymod_tvm::interface>(settings_.tvm.module) :
        nullptr;

    for (auto& nodes : settings_.nodes_by_shard)
    {
        bind_nodes(tvm, settings_.tvm.service, nodes);
        auto st = settings_.http;
        st.nodes = nodes;
        clients_.push_back(std::make_shared<yhttp::cluster_client>(*reactor_, st));
    }
}

yplatform::ptree hubrpc::get_stats() const
{
    yplatform::ptree ret;
    for (auto i = 0UL; i < clients_.size(); ++i)
    {
        ret.add_child("shard" + std::to_string(i + 1), clients_[i]->get_stats());
    }
    return ret;
}

size_t hubrpc::eval_shard(const string& sharding_key) const
{
    return sharding_key.size() ? gid_from_uid(sharding_key) * clients_.size() / NUM_GIDS :
                                 random_shard();
}

size_t hubrpc::random_shard() const
{
    static thread_local std::mt19937 generator(time(0));
    std::uniform_int_distribution<int> distribution(0, clients_.size() - 1);
    return distribution(generator);
}
}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::hubrpc)