#pragma once

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/module.h>
#include <yxiva/core/types.h>

namespace yxiva {

class hubrpc : public yplatform::module
{
public:
    using params_list = yhttp::params_list;
    using http_options = yhttp::cluster_client::options;
    struct settings
    {
        std::vector<std::vector<string>> nodes_by_shard;
        yhttp::cluster_client::settings http; // for all shards
        struct
        {
            string module;
            string service;
        } tvm;

        void parse_ptree(const yplatform::ptree& conf);
        void generate_nodes(const yplatform::ptree& conf);
    };

    hubrpc(yplatform::reactor& reactor, const yplatform::ptree& conf);

    // TODO support priority.

    template <typename Handler>
    void async_get(
        task_context_ptr ctx,
        const string& sharding_key,
        const string& path,
        params_list params,
        Handler&& handler);

    template <typename Handler>
    void async_get(
        task_context_ptr ctx,
        const string& sharding_key,
        const string& path,
        params_list params,
        http_options options,
        Handler&& handler);

    template <typename Handler>
    void async_post(
        task_context_ptr ctx,
        const string& sharding_key,
        const string& path,
        params_list params,
        string&& body,
        Handler&& handler);

    template <typename Handler>
    void async_post(
        task_context_ptr ctx,
        const string& sharding_key,
        const string& path,
        params_list params,
        string&& body,
        http_options options,
        Handler&& handler);

    template <typename Handler>
    void async_post_binary(
        task_context_ptr ctx,
        const string& sharding_key,
        const string& path,
        params_list params,
        string&& body,
        Handler&& handler);

    template <typename Handler>
    void async_post_binary(
        task_context_ptr ctx,
        const string& sharding_key,
        const string& path,
        params_list params,
        string&& body,
        http_options options,
        Handler&& handler);

    yplatform::ptree get_stats() const;

private:
    size_t eval_shard(const string& sharding_key) const;
    size_t random_shard() const;

    settings settings_;
    yplatform::reactor_ptr reactor_;
    std::vector<std::shared_ptr<yhttp::cluster_client>> clients_;
};

template <typename Handler>
inline void hubrpc::async_get(
    task_context_ptr ctx,
    const string& sharding_key,
    const string& path,
    params_list params,
    Handler&& handler)
{
    async_get(ctx, sharding_key, path, std::move(params), {}, std::forward<Handler>(handler));
}

template <typename Handler>
inline void hubrpc::async_post(
    task_context_ptr ctx,
    const string& sharding_key,
    const string& path,
    params_list params,
    string&& body,
    Handler&& handler)
{
    async_post(
        ctx,
        sharding_key,
        path,
        std::move(params),
        std::move(body),
        {},
        std::forward<Handler>(handler));
}

template <typename Handler>
inline void hubrpc::async_get(
    task_context_ptr ctx,
    const string& sharding_key,
    const string& path,
    params_list params,
    http_options options,
    Handler&& handler)
{
    auto shard = eval_shard(sharding_key);
    auto req = yhttp::request::GET(path + yhttp::url_encode(std::move(params)));
    clients_[shard]->async_run(
        ctx, std::move(req), std::move(options), std::forward<Handler>(handler));
}

template <typename Handler>
inline void hubrpc::async_post(
    task_context_ptr ctx,
    const string& sharding_key,
    const string& path,
    params_list params,
    string&& body,
    http_options options,
    Handler&& handler)
{
    auto shard = eval_shard(sharding_key);
    auto req = yhttp::request::POST(path + yhttp::url_encode(std::move(params)), std::move(body));
    clients_[shard]->async_run(
        ctx, std::move(req), std::move(options), std::forward<Handler>(handler));
}

template <typename Handler>
inline void hubrpc::async_post_binary(
    task_context_ptr ctx,
    const string& sharding_key,
    const string& path,
    params_list params,
    string&& body,
    Handler&& handler)
{
    async_post_binary(
        ctx,
        sharding_key,
        path,
        std::move(params),
        std::move(body),
        {},
        std::forward<Handler>(handler));
}

template <typename Handler>
inline void hubrpc::async_post_binary(
    task_context_ptr ctx,
    const string& sharding_key,
    const string& path,
    params_list params,
    string&& body,
    http_options options,
    Handler&& handler)
{
    auto shard = eval_shard(sharding_key);
    string headers = "Content-Type: application/octet-stream\r\n";
    auto req =
        yhttp::request::POST(path + yhttp::url_encode(std::move(params)), headers, std::move(body));
    clients_[shard]->async_run(
        ctx, std::move(req), std::move(options), std::forward<Handler>(handler));
}

} // namespace yxiva
