#include "xivaws_log.h"

namespace yxiva {

void xivaws_log::init(const yplatform::ptree& conf)
{
    parse_notification_fields(conf.get("log_notification_fields", ""));
    auto typed_log_id = conf.get("typed_log_id", "tskv");
    logger_ = yplatform::log::tskv_logger(YGLOBAL_LOG_SERVICE, typed_log_id);
    root_am_notifications << ll::make_attr("group", "notifications");
    root_am_subscriptions << ll::make_attr("group", "subscriptions");
    root_am_channels << ll::make_attr("group", "channels");
}

void xivaws_log::log_subscriber_add(
    const string& uid,
    const channel_key& key,
    const string& subscriber)
{
    YLOG_L(info) << "add " << subscriber << " to " << channel_full_name(key)
                 << " subscribers collection";

    TYPED_S << ll::make_attr("action", "add") << ll::make_attr("uid", uid) << make_attrs(key)
            << ll::make_attr("connection", subscriber);
}

void xivaws_log::log_subscriber_del(
    const string& uid,
    const channel_key& key,
    const string& subscriber)
{
    YLOG_L(info) << "del " << subscriber << " from " << channel_full_name(key)
                 << " subscribers collection";

    TYPED_S << ll::make_attr("action", "del") << ll::make_attr("uid", uid) << make_attrs(key)
            << ll::make_attr("connection", subscriber);
}

void xivaws_log::log_subscriber_del(
    const string& uid,
    const string& service,
    const string& subscriber)
{
    YLOG_L(info) << "del " << subscriber << " from {" << uid << "," << service
                 << "} subscribers collection";

    TYPED_S << ll::make_attr("action", "del") << ll::make_attr("uid", uid)
            << ll::make_attr("service", service) << ll::make_attr("connection", subscriber);
}

void xivaws_log::log_channel_state(
    const string& uid,
    const channel_key& key,
    const string& state,
    const string& subscription_id)
{
    YLOG_L(debug) << channel_full_name(key) << " state changed to " << state
                  << (subscription_id != "" ? " id=" + subscription_id : "");

    TYPED_CH << ll::make_attr("uid", uid) << ll::make_attr("status", state) << make_attrs(key)
             << ll::make_attr("subscription_id", subscription_id);
}

void xivaws_log::log_notification(
    task_context_ptr ctx,
    const string& subscription_id,
    /*const */ subscriber& subscriber,
    const message& message)
{
    std::stringstream ss;
    log_message(message, ss);
    YLOG_CTX_LOCAL(subscriber.ctx(), info)
        << "notification:" << ss.str() << ", context=\"" << ctx << "\""
        << ", client_id=\"" << subscriber.client_id() << "\""
        << ", subscription_id=\"" << subscription_id << "\"";

    TYPED_N << make_context_attr("send_context", ctx) << ll::make_attr("action", "send")
            << ll::make_attr("subscription_id", subscription_id)
            << ll::make_attr("connection", subscriber.ctx()->uniq_id()) << make_attrs(message);
}

void xivaws_log::log_notification(
    task_context_ptr send_ctx,
    task_context_ptr client_ctx,
    const message& message)
{
    TYPED_N << make_context_attr("send_context", send_ctx)
            << make_context_attr("subscribe_context", client_ctx) << ll::make_attr("action", "send")
            << make_attrs(message);
}

void xivaws_log::log_notification_skip(
    task_context_ptr ctx,
    const message& message,
    const string& info)
{
    TYPED_N << make_context_attr("send_context", ctx) << ll::make_attr("action", "skip")
            << ll::make_attr("info", info) << make_attrs(message);
}

void xivaws_log::log_notification_ack(
    task_context_ptr ctx,
    const string& send_context,
    const string& transit_id,
    const string& client,
    const string& service)
{
    YLOG_CTX_LOCAL(ctx, info) << "notification ack "
                              << " send_context=" << send_context << " transit_id=" << transit_id
                              << " client=" << client << " service=" << service;

    TYPED_N << make_context_attr("subscriber", ctx) << ll::make_attr("action", "ack")
            << ll::make_attr("send_context", send_context)
            << ll::make_attr("transit_id", transit_id) << ll::make_attr("client", client)
            << ll::make_attr("service", service);
}

void xivaws_log::parse_notification_fields(string const& log_notification_fields_str)
{
    if (!log_notification_fields_str.empty())
    {
        boost::split(
            log_notification_fields_,
            log_notification_fields_str,
            boost::is_any_of(","),
            boost::algorithm::token_compress_on);
    }
}

void xivaws_log::log_message(const message& message, std::stringstream& ss)
{
    string debug_print = find_all_map_values(message.data, log_notification_fields_, "");
    ss << " uid=\"" << message.uid << "\""
       << " service=\"" << message.service << "\""
       << " operation=\"" << message.operation << "\"";
    if (message.local_id != 0)
    {
        ss << " local_id=\"" << message.local_id << "\"";
    }
    ss << " transit_id=\"" << message.transit_id << "\""
       << (debug_print != "" ? " " + debug_print : "");
}

void xivaws_log::log_map_fields(const std::map<string, string>& map, ll::attributes_map& am)
{
    for (const auto& key : log_notification_fields_)
    {
        auto pair = map.find(key);
        am << ll::make_attr(key, pair != map.end() ? pair->second : "");
    }
}

ll::attributes_map xivaws_log::make_attrs(const channel_key& key)
{
    ll::attributes_map am;
    am << ll::make_attr("service", key.service) << ll::make_attr("client", key.client)
       << ll::make_attr("session", key.session) << ll::make_attr("filter", key.filter);
    return am;
}

ll::attributes_map xivaws_log::make_attrs(const message& message)
{
    ll::attributes_map am;
    am << ll::make_attr("uid", message.uid) << ll::make_attr("service", message.service)
       << ll::make_attr("event", message.operation);
    if (message.local_id != 0)
    {
        am << ll::make_attr("local_id", message.local_id);
    }
    am << ll::make_attr("transit_id", message.transit_id);
    log_map_fields(message.data, am);
    return am;
}

string xivaws_log::channel_full_name(const channel_key& key) const
{
    return "{" + static_cast<string>(key.service) + "," + string(key.filter) + "," + key.client +
        "," + key.session + "}";
}

void xivaws_log::log_rproxy_request(
    task_context_ptr ctx,
    const string& protocol_code,
    const string& service,
    uint64_t reqid,
    const string& private_error,
    size_t request_bytes,
    size_t response_bytes,
    int http_code,
    const time_duration& backend_time)
{
    TYPED << ll::make_attr("group", "rproxy") << ll::make_attr("event", "request")
          << ll::make_attr("protocol_code", protocol_code) << ll::make_attr("service", service)
          << ll::make_attr("reqid", reqid) << ll::make_attr("private_error", private_error)
          << ll::make_attr("http_code", http_code) << ll::make_attr("request_bytes", request_bytes)
          << ll::make_attr("response_bytes", response_bytes)
          << ll::make_attr("backend_time", time_traits::to_string(backend_time))
          << ll::make_attr("connection", ctx->uniq_id());
}

}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yxiva::xivaws_log)
