#include "catalogue.h"

#include <vector>
#include <yxiva/core/nested_prefix_stream.h>

namespace yxiva {

bool catalogue::add(
    const string& uid,
    const string& service,
    const string& subscription_id,
    subscriber_ptr subscriber)
{
    scoped_lock lock(mutex_);
    auto& by_user_service = subscribers_[{ uid, service }];
    auto [_, inserted] = by_user_service.insert({ subscription_id, subscriber });
    if (inserted)
    {
        stat_->add_subscriber(service);
    }
    return inserted;
}

bool catalogue::del(const string& uid, const string& service, const string& subscription_id)
{
    scoped_lock lock(mutex_);
    auto& by_user_service = subscribers_[{ uid, service }];
    auto deleted = by_user_service.erase(subscription_id);
    if (deleted)
    {
        stat_->del_subscriber(service);
    }
    return deleted;
}

std::size_t catalogue::del_by_subscriber_id(
    const string& uid,
    const string& service,
    const string& subscriber_id)
{
    std::size_t deleted = 0;
    scoped_lock lock(mutex_);
    auto by_uid_service = subscribers_.find({ uid, service });
    if (by_uid_service == subscribers_.end())
    {
        return deleted;
    }
    auto&& subscribers = by_uid_service->second;
    for (auto&& it = subscribers.begin(); it != subscribers.end();)
    {
        if (it->second->id() == subscriber_id)
        {
            it = subscribers.erase(it);
            stat_->del_subscriber(service);
            ++deleted;
        }
        else
        {
            ++it;
        }
    }
    return deleted;
}

std::vector<subscriber_ptr> catalogue::find_by_uid_service(const string& uid, const string& service)
{
    std::vector<subscriber_ptr> ret;
    scoped_lock lock(mutex_);
    auto it = subscribers_.find({ uid, service });
    if (it != subscribers_.end())
    {
        ret.reserve(it->second.size());
        for (auto&& [_, subscriber] : it->second)
        {
            ret.push_back(subscriber);
        }
    }
    return ret;
}

subscriber_ptr catalogue::find_by_id(
    const string& uid,
    const string& service,
    const string& subscription_id)
{
    scoped_lock lock(mutex_);
    auto it = subscribers_.find({ uid, service });
    if (it != subscribers_.end())
    {
        auto by_subscription_id = it->second.find(subscription_id);
        if (by_subscription_id != it->second.end())
        {
            return by_subscription_id->second;
        }
    }
    return {};
}

frontend_stat_ptr catalogue::stat()
{
    return stat_;
}

void catalogue::clear()
{
    scoped_lock lock(mutex_);
    subscribers_.clear();
}

void catalogue::write_overview(std::ostream& origin_stream)
{
    scoped_lock lock(mutex_);
    auto view_data = subscribers_;
    lock.unlock();

    nested_prefix_stream stream(origin_stream);
    stream << "Fields:\nuser channel filter session-key client-id subscription-id context\n\n";
    for (auto& [key, subscribers] : view_data)
    {
        stream.push_not_empty(static_cast<string>(key.first));
        stream.push_not_empty(static_cast<string>(key.second));
        for (auto& [subscription_id, subscriber] : subscribers)
        {
            stream.push_not_empty("");
            stream.push_not_empty("");
            stream.push_not_empty(subscriber->client_id());
            stream.push_not_empty(subscription_id);

            stream << (subscriber->ctx() ? subscriber->ctx()->uniq_id() : "-") << "\n";

            stream.pop();
            stream.pop();
            stream.pop();
            stream.pop();
        }
        stream.pop();
        stream.pop();
    }
}

}
