#pragma once

#include "gate.h"
#include "settings.h"
#include "mod_log/find.h"
#include <yxiva/core/binary_protocol.h>
#include <yplatform/module.h>
#include <yplatform/util/tuple_unpack.h>

namespace yxiva {

class rproxy : public yplatform::module
{
public:
    typedef yplatform::log::tskv_logger logger_type;

    rproxy(yplatform::reactor& reactor, const yplatform::ptree& ptree)
        : rproxy(reactor, create_rproxy_settings(ptree))
    {
    }

    rproxy(yplatform::reactor& reactor, const rproxy_settings& settings)
        : io_(*reactor.io()), settings_(settings)
    {
        for (auto& [backend, st] : settings.backends)
        {
            backends_[backend].client_settings = st.http;
            backends_[backend].client = std::make_shared<yhttp::cluster_client>(reactor, st.http);
            if (st.rate_limit)
            {
                backends_[backend].rate_limit = rproxy_rate_limit(*st.rate_limit);
            }
        }
    }

    template <typename... Args>
    void async_call(Args&&... args)
    {
        io_.post([self = yplatform::shared_from(this),
                  args = std::make_tuple(this, std::forward<Args>(args)...)]() mutable {
            using handler_type =
                std::tuple_element_t<std::tuple_size_v<decltype(args)> - 1, decltype(args)>;
            std::apply(&rproxy::do_async_call<handler_type>, std::move(args));
        });
    }

    bool enabled_for_any(const std::vector<string>& services)
    {
        return settings_.enabled_for_any(services);
    }

    // Updates only limits.
    void reload(const yplatform::ptree& ptree)
    {
        io_.post([this, self = yplatform::shared_from(this), ptree] {
            auto settings = create_rproxy_settings(ptree);
            settings_.max_frame_size = settings.max_frame_size;
            for (auto& [backend, gate] : backends_)
            {
                auto it = settings.backends.find(backend);
                if (it == settings.backends.end() || !it->second.rate_limit)
                {
                    gate.rate_limit.reset();
                }
                else
                {
                    gate.rate_limit = rproxy_rate_limit(*it->second.rate_limit);
                }
            }
        });
    }

    yplatform::ptree get_stats() const
    {
        yplatform::ptree backends_stats;
        for (auto&& [backend, gate] : backends_)
        {
            yplatform::ptree backend_stats = gate.client->get_stats();
            // Dirty read without synchronization.
            if (gate.stats.rate_limit.enabled &&
                clock::now() - gate.stats.update_ts < settings_.stats_ttl)
            {
                yplatform::ptree rate_limit_stats;
                rate_limit_stats.put("running", gate.stats.rate_limit.running);
                rate_limit_stats.put("limit", gate.stats.rate_limit.limit);
                backend_stats.put_child("rate_limit", rate_limit_stats);
            }
            backends_stats.put_child(backend, backend_stats);
        }
        yplatform::ptree ret;
        ret.put_child("backend", backends_stats);
        return ret;
    }

private:
    auto find_client(
        size_t backend_index,
        const std::vector<string>& whitelist,
        const string& custom_url)
    {
        std::shared_ptr<yhttp::cluster_client> client;
        string service;

        if (backend_index >= whitelist.size()) return std::tuple(client, service);

        if (auto it = backends_.find(whitelist[backend_index]); it != backends_.end())
        {
            client = it->second.client;
            service = whitelist[backend_index];
            if (settings_.allow_url_override && custom_url.size())
            {
                auto st = it->second.client_settings;
                st.nodes = { custom_url };
                client = std::make_shared<yhttp::cluster_client>(io_, st);
            }
        }

        return std::tuple(client, service);
    }

    template <typename Handler>
    void do_async_call(
        task_context_ptr ctx,
        const std::vector<string>& backend_whitelist,
        const string& custom_url,
        string headers,
        string data,
        Handler&& handler)
    {
        namespace bin = binary_protocol;
        using status_frame = bin::proxy_status_frame;
        using errc = bin::error_code;

        if (data.empty() || bin::unpack_type(data) != bin::type::data)
        {
            handle_error(
                ctx,
                handler,
                "",
                status_frame{ 0, errc::protocol_error },
                data.size(),
                "no data or invalid data type");
            return;
        }

        auto [unpacked, data_header] = bin::unpack_data_header(data);
        if (!unpacked)
        {
            handle_error(
                ctx,
                handler,
                "",
                status_frame{ 0, errc::corrupted_data_header },
                data.size(),
                "corrupted");
            return;
        }

        if (data.size() > settings_.max_frame_size)
        {
            handle_error(
                ctx,
                handler,
                "",
                status_frame{ data_header.reqid, errc::frame_too_large },
                data.size());
            return;
        }

        auto [client, service] =
            find_client(data_header.service_index, backend_whitelist, custom_url);
        if (!client)
        {
            handle_error(ctx, handler, "", status_frame{ 0, errc::backend_not_found }, data.size());
            return;
        }

        if (auto err = rate_limit_try_run(service))
        {
            handle_error(
                ctx,
                handler,
                service,
                status_frame{ data_header.reqid, err },
                data.size(),
                "rate limit exceeded");
            return;
        }

        make_stats_snapshot(service);
        size_t request_bytes = data.size();
        client->async_run(
            ctx,
            yhttp::request::POST(data_header.path, std::move(headers), std::move(data)),
            [this,
             self = shared_from(this),
             ctx,
             reqid = data_header.reqid,
             request_bytes,
             handler = std::forward<Handler>(handler),
             service = service,
             start_time = clock::now()](auto error, auto response) {
                if (!error && response.status == 200)
                {
                    find_xivaws_log()->log_rproxy_request(
                        ctx,
                        to_string(errc::success),
                        service,
                        reqid,
                        "",
                        request_bytes,
                        response.body.size(),
                        response.status,
                        clock::now() - start_time);
                    // XXX 'this->' prevent unused 'this' capture warning
                    this->invoke_handler(handler, std::move(response.body));
                }
                else
                {
                    handle_error(
                        ctx,
                        handler,
                        service,
                        status_frame{ reqid, errc::backend_call_error },
                        request_bytes,
                        error ? error.message() : "",
                        response.status);
                }
            });
    }

    template <typename Handler>
    void invoke_handler(Handler&& handler, string&& data)
    {
        io_.post([handler = std::forward<Handler>(handler), data = std::move(data)]() mutable {
            handler(std::move(data));
        });
    }

    template <typename Handler>
    void handle_error(
        task_context_ptr ctx,
        Handler&& handler,
        const string& service,
        binary_protocol::proxy_status_frame proxy_status,
        size_t request_length,
        const string& error_message = string(),
        int http_code = 0)
    {
        invoke_handler(std::forward<Handler>(handler), binary_protocol::pack(proxy_status));

        find_xivaws_log()->log_rproxy_request(
            ctx,
            to_string(proxy_status.error),
            service,
            proxy_status.reqid,
            error_message,
            request_length,
            0,
            http_code);
    }

    binary_protocol::error_code rate_limit_try_run(const string& service)
    {
        auto it = backends_.find(service);
        if (it == backends_.end() || !it->second.rate_limit) return binary_protocol::success;
        auto& rate_limit = *it->second.rate_limit;
        if (rate_limit.requests_counter.get() >= rate_limit.settings.limit)
            return rate_limit.settings.response_code;
        rate_limit.requests_counter.add(1);
        return binary_protocol::success;
    }

    void make_stats_snapshot(const string& service)
    {
        auto it = backends_.find(service);
        if (it == backends_.end()) return;
        auto& gate = it->second;
        auto& stats = gate.stats;
        stats.update_ts = clock::now();
        if (gate.rate_limit)
        {
            auto& rate_limit = *gate.rate_limit;
            stats.rate_limit.enabled = true;
            stats.rate_limit.running = rate_limit.requests_counter.get();
            stats.rate_limit.limit = rate_limit.settings.limit;
        }
        else
        {
            stats.rate_limit.enabled = false;
            stats.rate_limit.running = 0;
            stats.rate_limit.limit = 0;
        }
    }

    boost::asio::io_service& io_;
    rproxy_settings settings_;
    std::map<string, rproxy_gate> backends_;
};
}
